-- Blitz Basic file export

RegisterExporter("Blitz Basic (*.b3d)", "*.b3d", "b3d.export")
b3d = {}


function b3d.BuildMaterialTable()
   -- builds the material table, indexed by id, contains the new id used
   -- for export

   b3d.MaterialTable = {}

   local i
   local mat_counter = 0
   for i = 0, GetTextureCount()-1,1
      do
      local tex = GetTexture(i)
      b3d.MaterialTable[tex:GetId()] = mat_counter
      mat_counter = mat_counter + 1
      end

end


function b3d.StreamMaterials()
   -- reads all material information and streams it to the b3d exporter
   local i
   for i = 0, GetTextureCount()-1,1
      do
      local tex = GetTexture(i)
      local m = b3d_AddMaterial()
      b3d_MatSetName(m, "Material"..i)
      b3d_MatSetTexture(m, tex:GetAbsoluteFilename())
      end

end

function b3d.FindBestJoint(v)
   -- finds the joint with the highest influence on the vertex
   local i
   local best_joint = -1
   local best_weight = 0
   for i = 1,4,1
      do
      if (v.bone[i] > -1) then
         if (v.weight[i] > best_weight) then
            best_joint = v.bone[i]
            best_weight = v.weight[i]
            end
         end
      end
   return best_joint
end


function b3d.BuildJointTable()
   -- builds the joint table which which flattens the joint hierarchy
   -- indexed by joint id, contains the new id used for export
   b3d.JointTable = {}
   local scene = GetScene()
   local node = scene:GetFirstNode()
   b3d.joint_counter = 0
   while node
      do

      if (node:GetClassName() == "fxBone") then
         do
         local joint = tobone(node)
         b3d.JointTable[joint:GetId()] = b3d.joint_counter
         b3d.joint_counter = b3d.joint_counter + 1
         end
      end -- if

      node = scene:GetNextNode()
      end
   end


function b3d.ProcessNodes(node)

   -- process meshes
   if node:GetClassName() == "fxMesh" then

      local mesh = tomesh(node)

      local m = b3d_AddMesh()

      -- node name
      b3d_MeshSetName(m, mesh:GetName())

      -- material
      local mat_index = b3d.MaterialTable[mesh:GetMaterial()]
      if not mat_index then print("Material not found!") mat_index = -1 end
      b3d_MeshSetMaterial(m, mat_index)

      -- vertices
      local i
      for i = 0,mesh:GetVertexCount()-1,1
         do
         local vertex = mesh:GetLocalVertex(i)
--         local joint_index = b3d.JointTable[b3d.FindBestJoint(vertex)]

         local b1,b2,b3,b4
         local w1,w2,w3,w4

         b1 = b3d.JointTable[vertex.bone[1]]
         if not b1 then b1 = -1 end
         b2 = b3d.JointTable[vertex.bone[2]]
         if not b2 then b2 = -1 end
         b3 = b3d.JointTable[vertex.bone[3]]
         if not b3 then b3 = -1 end
         b4 = b3d.JointTable[vertex.bone[4]]
         if not b4 then b4 = -1 end

         w1 = vertex.weight[1]
         w2 = vertex.weight[2]
         w3 = vertex.weight[3]
         w4 = vertex.weight[4]

         if not joint_index then joint_index = -1 end
         b3d_MeshAddVertex(m,
                          vertex.position.x,
                          vertex.position.y,
                          vertex.position.z,
                          vertex.normal.x,
                          vertex.normal.y,
                          vertex.normal.z,
                          vertex.texcoords.x,
                          1-vertex.texcoords.y,
                          b1,
                          w1,
                          b2,
                          w2,
                          b3,
                          w3,
                          b4,
                          w4)
         end

      -- faces
      for i = 0, mesh:GetFaceCount()-1,1
         do
         local v1,v2,v3
         v1,v2,v3 = mesh:GetFace(i)
         b3d_MeshAddFace(m, v1,v2,v3)
         end

      end

   -- process bones
   if node:GetClassName() == "fxBone" then

      local bone = tobone(node)
      local b = b3d_AddBone()

      -- node name
      b3d_BoneSetName(b, bone:GetName())

      -- node parent
      local parent_bone = bone:GetParent()
      local parent_name = ""
      if parent_bone:GetClassName() == "fxBone" then parent_name = parent_bone:GetName() end
      b3d_BoneSetParentName(b, parent_name)

      -- initial (rest) position/rotation
      local matrix = bone:GetLocalMatrix()

      local pos = matrix:pos_component()
      local rot = matrix:get_eulers()

      -- set initial pos/rot
      b3d_BoneSetPosition(b, pos.x, pos.y, pos.z)
      b3d_BoneSetRotation(b, rot.x, rot.y, rot.z)

      -- additioanl provide the rotation as matrix, to avoid any problems
      local m1 = matrix:clone()
      m1:clear_translation()
      b3d_BoneSetMatrix(b, m1:M(0,0), m1:M(0,1), m1:M(0,2), m1:M(0,3),
                          m1:M(1,0), m1:M(1,1), m1:M(1,2), m1:M(1,3),
                          m1:M(2,0), m1:M(2,1), m1:M(2,2), m1:M(2,3),
                          m1:M(3,0), m1:M(3,1), m1:M(3,2), m1:M(3,3))
      m1:delete()

      -- position keys
      local i
      for i = 0,bone:GetKeyCount()-1, 1
         do
         key = bone:GetKey(i)

         local m = bone:GetLocalMatrix():clone()
         m:clear_translation()
         local m_inv = m:clone()
         m_inv:invert()

         m:mult_simple(key.matrix)
         m:mult_simple(m_inv);

         local pos = m:pos_component()

         m:delete()
         m_inv:delete()

         b3d_BoneAddPosKey(b, key.frame, pos.x, pos.y, pos.z)
         end

      -- rotation keys
      local i
      for i = 0,bone:GetKeyCount()-1, 1
         do
         key = bone:GetKey(i)

         local m = bone:GetLocalMatrix():clone()
         m:clear_translation()
         local m_inv = m:clone()
         m_inv:invert()

         m:mult_simple(key.matrix)
         m:mult_simple(m_inv);

         local rot = m:get_eulers()

         m:delete()
         m_inv:delete()

         b3d_BoneAddRotKey(b, key.frame, rot.x, rot.y, rot.z)
         end

      end

end



function b3d.export(filename)

   filename = filename..".b3d"

   -- preprocess scene
   b3d.BuildJointTable()
   b3d.BuildMaterialTable()

   -- stream scene nodes to exporter
   b3d_BeginModel()

   b3d_SetTotalFrames(GetTotalFrames())
   b3d_SetFramerate(GetFrameRate())

   -- stream material info to exporter
   b3d.StreamMaterials()

   IterateNodes(b3d.ProcessNodes)

   -- get a pointer to the model
   model = b3d_GetModel()
   assert(model)

   -- execute the plugin to save the model to a file
   plug = CreatePlugin_b3dExport()
   plug:Execute(model, filename)
   plug:delete()

   b3d_EndModel()

end
