
-- Nebula export

RegisterExporter("Nebula Script (*.n)", "*.n", "Nebula.Export")
Nebula = {}



-------------------------------------------------------------------------------
-- text to write at the beginning of the file
function Nebula.WriteProlog()
write(Nebula.filehandle, "sel /usr/scene\n")

write(Nebula.filehandle, "new n3dnode scene\n")
write(Nebula.filehandle, "   sel scene\n")
end


-------------------------------------------------------------------------------
-- text to write at the end of the file

function Nebula.WriteEpilog()
write(Nebula.filehandle, "sel ..\n")
write(Nebula.filehandle, "/sys/servers/channel.setchannel1f default 1\n")
end


-------------------------------------------------------------------------------
-- collect all root joints and store their ids in Nebula.RootJoints
function Nebula.CollectRootJoints()
Nebula.RootJoints = {}
Nebula.RootJointCount = 0

local scene = GetScene()
local node = scene:GetRootNode()
local i

for i = 0, node:GetChildCount()-1,1
	do
	local child_node = node:GetChild(i)
	if child_node:GetClassName() == "fxBone" then
		do
		Nebula.RootJointCount = Nebula.RootJointCount + 1
		Nebula.RootJoints[Nebula.RootJointCount] = tobone(child_node):GetId()
		end
	end -- if

	end
end



-------------------------------------------------------------------------------
-- text to write before a character is exported
function Nebula.SaveCharacterProlog(number)
write(Nebula.filehandle, "   new n3dnode character",number,"\n")
write(Nebula.filehandle, "      sel character",number,"\n")
end


-------------------------------------------------------------------------------
-- text to write after a character has been exported
function Nebula.SaveCharacterEpilog()
write(Nebula.filehandle, "   sel ..\n")
end



-------------------------------------------------------------------------------
-- saves the njoint node hierarchy
function Nebula.SaveJointHierarchy(joint_num, spaces)
local bone = FindJoint(joint_num)
assert(bone)

if not spaces then spaces = "      " else spaces = spaces.."   " end

write(Nebula.filehandle, spaces, 'new njoint "',bone:GetName(),'"\n')
write(Nebula.filehandle, spaces, '   sel "', bone:GetName(), '"\n')
write(Nebula.filehandle, spaces, "   .setactive true\n")

local matrix = bone:GetLocalMatrix()
local matrix_pos = matrix:pos_component()
local matrix_rot = matrix:get_quaternion()
matrix_rot:invert()

write(Nebula.filehandle, spaces, "   .nxyz ", matrix_pos.x, " ", matrix_pos.y, " ", matrix_pos.z, "\n")
write(Nebula.filehandle, spaces, "   .oqxyzw ", matrix_rot.x, " ", matrix_rot.y, " ", matrix_rot.z, " ", matrix_rot.w, "\n")

-- save children
local i = 0
for i = 0, bone:GetChildCount()-1,1
	do
	local child = bone:GetChild(i)
	if child:GetClassName() == "fxBone" then
		do
		local child_bone = tobone(child)
		Nebula.SaveJointHierarchy(child_bone:GetId(), spaces)
		end
	end -- if
	end

write(Nebula.filehandle, spaces, "sel ..\n")

end

-------------------------------------------------------------------------------
-- builds the "full" name of a node, that is a name that incorporates the
-- parent's names
function Nebula.GetFullName(bone)

local result = bone:GetName()

local parent_node = bone:GetParent()

while parent_node
        do
        if parent_node:GetClassName() == "fxBone" then
                do
                local temp_result = "/"..result
                local parent_as_bone = tobone(parent_node)
                assert(parent_as_bone)
                result = parent_as_bone:GetName()..temp_result
                end
           else return result end
        parent_node = parent_node:GetParent()
        end

return result
end

-------------------------------------------------------------------------------
-- builds a table which contains all joints that belong to joint_num
function Nebula.BuildJointTable(joint_num)
local bone = FindJoint(joint_num)
assert(bone)

Nebula.JointCount = Nebula.JointCount + 1
Nebula.JointTable[Nebula.JointCount] = {id = bone:GetId(), name = bone:GetName(), fullname = Nebula.GetFullName(bone) }

-- now visit our kids
local i
for i = 0, bone:GetChildCount()-1,1
	do
	local child = bone:GetChild(i)
	if child:GetClassName() == "fxBone" then
		do
		local child_bone = tobone(child)
		Nebula.BuildJointTable(child_bone:GetId())
		end
	end -- if
	end

end

-------------------------------------------------------------------------------
--
function Nebula.BuildKeyTable(table, joint_num)
local bone = FindJoint(joint_num)
assert(bone)

Nebula.KeyCounter = Nebula.KeyCounter + 1
local matrix = bone:GetOffsetMatrix()
table[Nebula.KeyCounter] = {pos = matrix:pos_component(), rot = matrix:get_eulers(), rel = bone:GetLocalMatrix(), ofs = bone:GetOffsetMatrix() }

-- now visit our kids
local i
for i = 0, bone:GetChildCount()-1,1
	do
	local child = bone:GetChild(i)
	if child:GetClassName() == "fxBone" then
		do
		local child_bone = tobone(child)
		Nebula.BuildKeyTable(table, child_bone:GetId())
		end
	end -- if
	end
end

-------------------------------------------------------------------------------
-- saves the njointanim nodes
function Nebula.SaveJointAnimation(joint_num)

Nebula.JointTable = {}
Nebula.JointCount = 0

Nebula.BuildJointTable(joint_num)

write(Nebula.filehandle, "      new njointanim anim\n")
write(Nebula.filehandle, "         sel anim\n")

write(Nebula.filehandle, '         .setreptype "loop"\n')
write(Nebula.filehandle, '         .setchannel "time"\n')
write(Nebula.filehandle, '         .setscale 1.0\n')

write(Nebula.filehandle, "         .beginjoints ", Nebula.JointCount,"\n")

local i
for i=1,Nebula.JointCount,1
	do
	write(Nebula.filehandle, "         .setjoint ", i-1, ' "', Nebula.JointTable[i].name, '" "',Nebula.JointTable[i].fullname,'" true true\n')
	end

write(Nebula.filehandle,"          .endjoints\n")

write(Nebula.filehandle,"         .beginstates 1\n")
write(Nebula.filehandle,'         .setstate 0 "default" ', GetTotalFrames(), "\n")
write(Nebula.filehandle,"         .endstates\n")

Nebula.KeyTable = {}

for i = 1,GetTotalFrames(),1
        do
        ProgressWindowSet(i)
        SetFrame(i)
        Nebula.KeyTable[i] = {}
        Nebula.KeyCounter = 0
        Nebula.BuildKeyTable(Nebula.KeyTable[i], joint_num)
        end

write(Nebula.filehandle,"         .beginkeys ",1 / GetFrameRate(),"\n")

for i = 1, Nebula.JointCount,1
        do
	write(Nebula.filehandle, '         .begintranslate "',Nebula.JointTable[i].name,'" default\n')
        local j
        for j = 1, GetTotalFrames(),1
                do
                local jt = Nebula.KeyTable[j]
		write(Nebula.filehandle, "         .setkey ", j-1, " ", jt[i].pos.x, " ", jt[i].pos.y, " ", jt[i].pos.z, "\n")
                end
	write(Nebula.filehandle, '         .endtranslate\n')
	write(Nebula.filehandle, '         .beginrotate "',Nebula.JointTable[i].name,'" default\n')
        for j = 1, GetTotalFrames(),1
                do
                local jt = Nebula.KeyTable[j]

                m = jt[i].rel:clone()

                m:rotate_x(jt[i].rot.x)
                m:rotate_y(jt[i].rot.y)
                m:rotate_z(jt[i].rot.z)

                m_inv = jt[i].rel:clone()
                m_inv:invert()

                m:mult_simple(m_inv)

                rot = m:get_eulers()

--		write(Nebula.filehandle, "         .setkey ", j-1, " ", rad2deg(jt[i].rot.x), " ", rad2deg(jt[i].rot.y), " ", rad2deg(jt[i].rot.z), "\n")
		write(Nebula.filehandle, "         .setkey ", j-1, " ", rad2deg(rot.x), " ", rad2deg(rot.y), " ", rad2deg(rot.z), "\n")
                m:delete()
                m_inv:delete()
                end
	write(Nebula.filehandle, '         .endrotate\n')
        end

write(Nebula.filehandle,"         .endkeys\n")
write(Nebula.filehandle,"      sel ..\n")
end



-------------------------------------------------------------------------------
-- Save the nmeshnodes and stuff
function Nebula.SaveMeshClusters(joint_num)
local bone
if joint_num then bone = FindJoint(Nebula.JointTable[joint_num].id) assert(bone) end

local mesh_num = 0
local scene = GetScene()
local node = scene:GetFirstNode()
while(node)
	do
	if node:GetClassName() == "fxMesh" then
		do
		mesh_num = mesh_num + 1

		write(Nebula.filehandle, "   new n3dnode mesh",mesh_num, "\n")
		write(Nebula.filehandle, "      sel mesh", mesh_num, "\n")

		local matrix = node:GetMatrix()

		if joint_num then
			write(Nebula.filehandle, "      new nmeshcluster meshcluster", mesh_num, "\n")
			write(Nebula.filehandle, "         sel meshcluster", mesh_num, "\n")
		end

		write(Nebula.filehandle, "         new nmeshnode meshnode", mesh_num, "\n")
		write(Nebula.filehandle, "            sel meshnode", mesh_num, "\n")
		write(Nebula.filehandle, "            .setfilename mesh", mesh_num, ".n3d\n")
		
		if joint_num then
			write(Nebula.filehandle, "            .setreadonly true\n")
		else
			write(Nebula.filehandle, "            .setreadonly false\n")
		end
		
		write(Nebula.filehandle, "         sel ..\n")

		if joint_num then
			write(Nebula.filehandle, "         .setskinmesh meshnode", mesh_num, "\n")
			write(Nebula.filehandle, '         .setrootjoint "../../',bone:GetName(), '"\n')
			write(Nebula.filehandle, "      sel ..\n")
		end

		write(Nebula.filehandle,"      new nshadernode shader\n")
		write(Nebula.filehandle,"         sel shader\n")
		write(Nebula.filehandle,"         .setnumstages 1\n")
		write(Nebula.filehandle,'         .setcolorop 0 "mul tex prev"\n')
		write(Nebula.filehandle,"         .begintunit 0\n")
		write(Nebula.filehandle,"            .setaddress wrap wrap\n")
		write(Nebula.filehandle,"            .setminmagfilter linear linear\n")
		write(Nebula.filehandle,"            .settexcoordsrc uv0\n")
		write(Nebula.filehandle,"            .setenabletransform false\n")
		write(Nebula.filehandle,"         .endtunit\n")
		write(Nebula.filehandle,"         .setlightenable false\n")
		write(Nebula.filehandle,"         .setalphaenable false\n")
		write(Nebula.filehandle,"         .setdiffuse  0.800000 0.800000 0.800000 1.000000\n")
		write(Nebula.filehandle,"         .setambient  0.200000 0.200000 0.200000 1.000000\n")
		write(Nebula.filehandle,"         .setemissive 0.000000 0.000000 0.000000 1.000000\n")
		write(Nebula.filehandle,"      sel ..\n")

		local tex = FindMaterial(node:GetMaterial())
		if (tex) then
			do
			write(Nebula.filehandle, "     new ntexarraynode tex\n")
			write(Nebula.filehandle, '     tex.settexture 0 "', tex:GetFileName() ,'" none\n')
			end
		end


		write(Nebula.filehandle, "   sel ..\n")

		Nebula.SaveMeshFile(tomesh(node), mesh_num, bone_num)

		end
	end
	node = scene:GetNextNode()
	end
end



-------------------------------------------------------------------------------
-- get the index of the joint given by joint_num
function Nebula.GetJointIndex(joint_num)
local i
for i = 1, Nebula.JointCount, 1
        do
        if Nebula.JointTable[i].id == joint_num then return (i-1) end
        end
return 0
end



-------------------------------------------------------------------------------
-- saves one mesh to a file
function Nebula.SaveMeshFile(mesh, mesh_num)
local mesh_file_name = format("%s%i%s", "mesh", mesh_num, ".n3d")
local meshhandle = openfile(mesh_file_name, "wt")
assert(meshhandle)

mesh:NormalizeWeights()

matrix = mesh:GetMatrix()
nmatrix = matrix44:new()

local matrix_temp_i
local matrix_temp_j

for matrix_temp_i=0,3,1
	do
	for matrix_temp_j=0,3,1
		do
		nmatrix:S(matrix_temp_i, matrix_temp_j, matrix:M(matrix_temp_i, matrix_temp_j))
		end
	end

-- reset translation
nmatrix:S(3,0,0)
nmatrix:S(3,1,0)
nmatrix:S(3,2,0)

local no_joints = 1
local has_joints = 1
local i
for i = 0, mesh:GetVertexCount()-1,1
	do
	local v = mesh:GetVertex(i)
	-- apply the modelview matrix to the vertices
	local vp = matrix:Transform(v.position)
	local vn = nmatrix:Transform(v.normal)

	write(meshhandle, "v " , vp.x, " ", vp.y, " ", vp.z, "\n")

        local bone_t
        local num_joints = 0
        for bone_t = 1,4,1
                do
                if v.bone[bone_t] >= 0 then num_joints = num_joints + 1 end
                end

	if num_joints == 0 then no_joints = 2 end
        if num_joints == 1 then write(meshhandle, "jw1") has_joints = 2 end
        if num_joints == 2 then write(meshhandle, "jw2") has_joints = 2 end
        if num_joints == 3 then write(meshhandle, "jw3") has_joints = 2 end
        if num_joints == 4 then write(meshhandle, "jw4") has_joints = 2 end

        for bone_t = 1,4,1
                do
                if v.bone[bone_t] >= 0 then
                        do         
		        write(meshhandle, " ", Nebula.GetJointIndex(v.bone[bone_t]), " ")
                        if v.weight[bone_t] == 1 then write(meshhandle, "1.0") else write(meshhandle, v.weight[bone_t]) end
                        end
                end
                end
        write(meshhandle, "\n")
--	write(meshhandle, "jw1 ",Nebula.GetJointIndex(v.bone[1]), " 1.0 \n")
	write(meshhandle, "vn ", vn.x, " ", vn.y, " ", vn.z, "\n")
	write(meshhandle, "vt ", v.texcoords.x, " ", v.texcoords.y, "\n")
	end

if (has_joints == 2) and (no_joints == 2) then error("Some vertices are assigned to joints and some not! They must be either all assigned or all not assigned!") end

for i = 0, mesh:GetFaceCount()-1,1
	do
	local f1, f2, f3 = mesh:GetFace(i)
	f1 = f1+1
	f2 = f2+1
	f3 = f3+1
	write(meshhandle, "f ",f1,"/",f1,"/",f1, " ", f2,"/",f2,"/",f2, " ", f3,"/",f3,"/",f3, "\n")
	end

closefile(meshhandle)
nmatrix:delete()
end


-------------------------------------------------------------------------------
-- first sets all global variables to nil and then manually runs the garbage
-- collector -> this frees the used data
function Nebula.FreeData()

Nebula.RootJoints = nil
Nebula.RootJointCount = nil

Nebula.JointTable = nil
Nebula.JointCount = nil

Nebula.KeyTable = nil

collectgarbage()
end


-------------------------------------------------------------------------------
-- main exporter function
function Nebula.Export(filename)

-- open the file
filename = filename..".n"

Nebula.filehandle = openfile(filename, "wt")
assert(Nebula.filehandle)

ProgressWindowMinMax(1, GetTotalFrames());
ProgressWindowShow()

Nebula.WriteProlog()

Nebula.CollectRootJoints()

local i
if Nebula.RootJointCount > 0 then
for i = 1, Nebula.RootJointCount, 1
	do
        Nebula.SaveCharacterProlog(i)
	Nebula.SaveJointHierarchy(Nebula.RootJoints[i])
	Nebula.SaveJointAnimation(Nebula.RootJoints[i])
        Nebula.SaveMeshClusters(i)
        Nebula.SaveCharacterEpilog()
	end
else
	do
        Nebula.SaveMeshClusters(nil)
	end
end

Nebula.WriteEpilog()

closefile(Nebula.filehandle)

Nebula.FreeData()

ProgressWindowHide()

end
