#version 120

#define PI 3.14159265

uniform samplerCube envMap;
uniform mat4 viewMatrix;

uniform sampler2D albedoMap;
uniform sampler2D normalMap;
uniform sampler2D roughnessMap;
uniform sampler2D metallicMap;
uniform sampler2D heightMap;
uniform sampler2DShadow shadowMap;

uniform bool hasAlbedoMap;
uniform bool hasNormalMap;
uniform bool hasRoughnessMap;
uniform bool hasMetallicMap;
uniform bool hasHeightMap;

const vec2 shadowMapSize = vec2(1024.0, 1024.0);
const float shadowBlurRadius = 2.0;

varying vec3 n;
varying vec3 position;
varying vec3 worldView;
varying vec3 e;
varying vec4 shadowCoord;

const float parallaxScale = 0.03;
const float parallaxBias = -0.01;

mat3 cotangentFrame(vec3 N, vec3 p, vec2 uv)
{
    vec3 dp1 = dFdx(p);
    vec3 dp2 = dFdy(p);
    vec2 duv1 = dFdx(uv);
    vec2 duv2 = dFdy(uv);
    vec3 dp2perp = cross(dp2, N);
    vec3 dp1perp = cross(N, dp1);
    vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;
    vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;
    float invmax = inversesqrt(max(dot(T, T), dot(B, B)));
    return mat3(T * invmax, B * invmax, N);
}

float beckmann(float NH, float roughness)
{
    float NH_sq = NH * NH;
    float NH_sq_r = 1.0 / (NH_sq * roughness * roughness);
    float roughness_exp = (NH_sq - 1.0) * NH_sq_r;
    return exp(roughness_exp)*NH_sq_r/(4.0*NH_sq);
}
    
float cookTorrance(
    vec3 lightDirection,
    vec3 viewDirection,
    vec3 surfaceNormal,
    float roughness,
    float fresnel)
{
    float VdotN = max(dot(viewDirection, surfaceNormal), 0.0);
    float LdotN = max(dot(lightDirection, surfaceNormal), 0.0);
    vec3 H = normalize(lightDirection + viewDirection);
    float NdotH = max(dot(surfaceNormal, H), 0.0);
    float VdotH = max(dot(viewDirection, H), 0.000001);
    float LdotH = max(dot(lightDirection, H), 0.000001);
    float G = 2.0 * NdotH / max(VdotH, 0.000001);
    G = min(1.0, G * min(VdotN, LdotN));
    float D = beckmann(NdotH, roughness);
    float F = fresnel + pow(1.0 - VdotH, 5.0) * (1.0 - fresnel);
    return  G * F * D / max(PI * VdotN, 0.000001);
}

float shadowLookup(sampler2DShadow depths, vec4 coord, vec2 offset)
{
    vec2 texelSize = 1.0 / shadowMapSize;
    vec2 v = offset * texelSize * coord.w;
    float z = shadow2DProj(depths, coord + vec4(v.x, v.y, 0.0, 0.0)).z;
    return z;
}

void main(void)
{
    vec2 texCoords = gl_TexCoord[0].st;
    vec3 nn = normalize(n);    
    vec3 N = nn;
    vec3 E = normalize(e);
    
    if (hasNormalMap)
    {
        mat3 TBN = cotangentFrame(nn, -e, texCoords);
        
        if (hasHeightMap)
        {
            vec2 eye2 = normalize(e * TBN).xy;
            float height = texture2D(heightMap, texCoords).r; 
            height = height * parallaxScale + parallaxBias;
            texCoords = texCoords + (height * eye2);
        }

        vec3 tN = normalize(texture2D(normalMap, texCoords).rgb * 2.0 - 1.0);
        N = normalize(TBN * tN);
    }
    
    vec3 worldN = N.xyz * mat3(viewMatrix);
    vec3 worldR = reflect(normalize(worldView), worldN);
    worldR.x = -worldR.x;
    
    float shadow = 1.0;
    if (shadowCoord.w > 0.0)
    {
        shadow = 0.0;
        if (shadowBlurRadius > 0.0)
        {
            float x, y;
	        for (y = -shadowBlurRadius ; y < shadowBlurRadius ; y += 1.0)
	        for (x = -shadowBlurRadius ; x < shadowBlurRadius ; x += 1.0)
            {
	            shadow += shadowLookup(shadowMap, shadowCoord, vec2(x, y));
            }
	        shadow /= shadowBlurRadius * shadowBlurRadius * 4.0;
        }
        else
            shadow = shadowLookup(shadowMap, shadowCoord, vec2(0.0, 0.0));
    }
    
    vec4 albedo = hasAlbedoMap? texture2D(albedoMap, texCoords) : gl_FrontMaterial.diffuse;
    
    float specularity = 1.0;
    float roughnessParam = 0.2;
    float metallicParam = 0.0;
    float roughness = hasRoughnessMap? texture2D(roughnessMap, texCoords).r : roughnessParam;
    roughness = mix(0.0, 1.0, roughness);
    float metallic = hasMetallicMap? texture2D(metallicMap, texCoords).r : metallicParam;
    metallic = mix(0.001, 1.0, metallic);
    
    int diffLod = int(8.0);
    
    int specMax = int(ceil(roughness));
    int specMin = int(floor(roughness));
    float specLodA = roughness - float(specMin);
    
    vec4 ambTex = textureCubeLod(envMap, worldN, diffLod);
    vec4 ambTexSpec = mix(
        textureCubeLod(envMap, worldR, specMin * 8), 
        textureCubeLod(envMap, worldR, specMax * 8), specLodA);
    
    vec3 positionToLightSource;
    vec3 L;
    float distanceToLight;
    float attenuation = 1.0;
    vec4 Ld, Ls;
    
    vec4 col_d = vec4(0.0, 0.0, 0.0, 0.0);
    vec4 col_s = vec4(0.0, 0.0, 0.0, 0.0);
    
    // TODO: make this uniform
    const int maxNumLights = 1;
    
    for (int i = 0; i < maxNumLights; i++)
    {
        if (gl_LightSource[i].position.w > 0.0)
        {
            positionToLightSource = vec3(gl_LightSource[i].position.xyz - position);
            distanceToLight = length(positionToLightSource);
            L = normalize(positionToLightSource);
            attenuation = 
                gl_LightSource[i].constantAttenuation / 
          ((1.0+gl_LightSource[i].linearAttenuation * distanceToLight) * 
           (1.0+gl_LightSource[i].quadraticAttenuation * distanceToLight * distanceToLight));
        }
        else
        {
            L = gl_LightSource[i].position.xyz;
            attenuation = 1.0;
        }

        Ld = gl_LightSource[i].diffuse; 
        Ls = gl_LightSource[i].specular;
    
        col_d += Ld * clamp(dot(N, L), 0.0, 1.0) * attenuation;
        col_s += Ls * clamp(cookTorrance(L, E, N, roughness, 0.3), 0.0, 1.0) * attenuation;
    }
    
    float NE = max(0.0, dot(N, E));
    
    float fresnel = (1.0 - roughness) * pow(1.0 - NE, 5.0);
    
    vec4 diffuseLight = albedo * (ambTex + col_d * shadow);
    vec4 directReflectLight = (albedo * (ambTexSpec)) * specularity;
    vec4 directLight = diffuseLight * (1.0 - metallic) + directReflectLight * metallic + col_s * shadow;
    vec4 totalLight = directLight * (1.0 - fresnel) + ambTexSpec * fresnel * specularity * shadow;
        
    gl_FragColor = totalLight;
    gl_FragColor.a = albedo.a;
}
