#include <stdio.h>
#include <windows.h>
#include "include/xtreme3d.h"
#include "include/engine.h"
#include "include/viewer.h"
#include "include/dummycube.h"
#include "include/camera.h"
#include "include/light.h"
#include "include/fonttext.h"
#include "include/primitives.h"
#include "include/object.h"
#include "include/material.h"
#include "include/mouse.h"
#include "include/terrain.h"

LRESULT CALLBACK WndProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);  

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int iCmdShow)
{
    WNDCLASS wc;
    HWND hwnd;
    HDC hDC;
    HGLRC hRC;        
    MSG msg;
    BOOL bQuit = FALSE;
    float theta = 0.0f;

    /* Register window class */
    wc.style = CS_OWNDC;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = "Xtreme3D";
    RegisterClass (&wc);

    /* Screen options */
    int s_real_width = GetSystemMetrics(SM_CXSCREEN);
    int s_real_height = GetSystemMetrics(SM_CYSCREEN);
    int s_width = 800;
    int s_height = 600;
    int s_depth = 32;
    BOOL s_fullscreen = false;

    /* Create main window */
    if (s_fullscreen) 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Terrain Demo", 
      WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
      SetWindowPos(hwnd,HWND_TOP,0,0,s_width,s_height,SWP_NOSIZE);
    }
    else 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Terrain Demo", 
      WS_CAPTION | WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
    }

    DEVMODE sDevMode;
    ZeroMemory(&sDevMode, sizeof(DEVMODE));
    sDevMode.dmSize = sizeof(DEVMODE);
    sDevMode.dmPelsWidth = s_width;
    sDevMode.dmFields |= DM_PELSWIDTH;
    sDevMode.dmPelsHeight = s_height;
    sDevMode.dmFields |= DM_PELSHEIGHT;
    sDevMode.dmBitsPerPel = s_depth;
    sDevMode.dmFields |= DM_BITSPERPEL;
    
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);
    
    /* Xtreme3D functions */     
    EngineCreate(double(int(hwnd)));        
    MouseShowCursor(0);    
    
    double matlib = MaterialLibraryCreate();
    MaterialLibraryActivate(matlib); 
    
    double view1 = ViewerCreate(0,0,s_width,s_height);  
    ViewerSetLighting(view1,0);
    ViewerSetBackgroundColor(view1,c_white);
    ViewerEnableFog(view1,1);
    ViewerSetFogColor(view1,c_white);
    ViewerSetFogDistance(view1,300,800);
    ViewerEnableVSync(view1,vsmSync);
    
    //RGB(0,0,255)
    
    double back = DummycubeCreate(0);
    double scene = DummycubeCreate(0); 
    double front = DummycubeCreate(0);
    
    double light = LightCreate(lsOmni,0);
    ObjectSetPosition(light,5,40,5);
        
    double shadows = DummycubeCreate(scene);
    //double sp = ShadowplaneCreate(64,64, 8,8, shadows, light, c_black, 0.6, scene);
    //ObjectSetMaterial(sp,"mfloor");
    //ObjectPitch(sp,90);
    
    //double cube = CubeCreate(5,5,5,shadows);
    //ObjectSetPosition(cube,0,2.5,-5);
    //ObjectSetMaterial(cube,"mfloor");
    
    MaterialCreate("mHDS","media/terrain_8bit.bmp",0);
    double hds = BmpHDSCreate("mHDS");
    double terrain = TerrainCreate(scene);
    TerrainSetHeightData(terrain,hds);
    TerrainSetTileSize(terrain,32);
    TerrainSetTilesPerTexture(terrain,8);
    TerrainSetQualityDistance(terrain,150);
    TerrainSetQualityStyle(terrain,0);
    TerrainSetMaxCLodTriangles(terrain,10000);
    TerrainSetOcclusionFrameSkip(terrain,0);
    TerrainSetOcclusionTesselate(terrain,1);
    ObjectSetScale(terrain,1,1,0.1);
    ObjectRotate(terrain,90,0,0);

    MaterialCreate("mTerrain","media/snow.jpg",0);
    MaterialCreate("detmap","media/detmap.jpg",0);
    MaterialSetTextureScale("detmap",32,32);
    MaterialSetBlendingMode("detmap",bmTransparency);
    MaterialSetSecondTexture("mTerrain","detmap");
    MaterialSetOptions("mTerrain",0,1);
    MaterialSetOptions("mTerrain",0,1);
    ObjectSetMaterial(terrain,"mTerrain");
    
    double camPos = DummycubeCreate(scene);
    ObjectSetPosition(camPos,0,50,40);
    
    double camera = CameraCreate(camPos);
    CameraSetViewDepth(camera,800);
    CameraSetFocal(camera,120);
    CameraSetSceneScale(camera,0.5);
    ViewerSetCamera(view1,camera);
    
    double font = WindowsBitmapfontCreate("Arial",10);
    double text_fps = HUDTextCreate(font,"FPS: ",front);
    TextSetColor(text_fps,c_black,0.6);
    ObjectSetPosition(text_fps,16,8,0);
    
    double deltax = 0;
    double deltay = 1;
    double pit = 0;
    double mx = 320;
    double my = 240;
    
    MouseSetPosition(mx,my);
    
    static char FPS[256];

    /* Program main loop */
    while (!bQuit)
    {
        /* Check for messages */
        if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        {
            if (msg.message == WM_QUIT)
            {
                bQuit = TRUE;
            }
            else
            {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
            }
        }
        else
        {
            /* Xtreme3D functions */
            if (GetKeyState('W')      & 0x80) ObjectMove(camPos,  -0.5);
            if (GetKeyState('S')      & 0x80) ObjectMove(camPos,   0.5);
            if (GetKeyState('A')      & 0x80) ObjectStrafe(camPos, 0.5);
            if (GetKeyState('D')      & 0x80) ObjectStrafe(camPos,-0.5);
            
            ObjectSetPositionY(camPos,TerrainGetHeightAtObjectPosition(terrain,camPos)+4);
            
            deltax = (mx-MouseGetX())/6;
            deltay = (my-MouseGetY())/6;
            ObjectRotate(camPos,0,deltax,0);
            pit = ObjectGetPitch(camera)+deltay;
            if (pit<89 || pit>-89) ObjectRotate(camera,deltay,0,0);
            MouseSetPosition(mx,my);
            
            //Update();
            ViewerRender(view1);
            
            sprintf(FPS, "FPS = %.f", GetFPS() );
            TextSetText(text_fps, FPS);
            Sleep(2);
        }
    }

    EngineDestroy();
    FreeLibrary(hLibXtreme3D);
    
    DestroyWindow (hwnd);

    sDevMode.dmPelsWidth = s_real_width;
    sDevMode.dmPelsHeight = s_real_height;
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);

    return msg.wParam;
}


LRESULT CALLBACK WndProc (HWND hWnd, UINT message,
                          WPARAM wParam, LPARAM lParam)
{

    switch (message)
    {
    case WM_CREATE:
        return 0;
        
    case WM_CLOSE:
        PostQuitMessage (0);
        return 0;

    case WM_DESTROY:
        return 0;

    case WM_KEYDOWN:
        switch (wParam)
        {
        case VK_ESCAPE:
            {
            PostQuitMessage(0);
            return 0;
            }
        }
        return 0;

    default:
        return DefWindowProc (hWnd, message, wParam, lParam);
    }
}
