#include <stdio.h>
#include <windows.h>
#include "include/xtreme3d.h"
#include "include/engine.h"
#include "include/viewer.h"
#include "include/dummycube.h"
#include "include/camera.h"
#include "include/light.h"
#include "include/fonttext.h"
#include "include/primitives.h"
#include "include/object.h"
#include "include/material.h"
#include "include/mouse.h"
#include "include/shadowplane.h"
#include "include/actor.h"

LRESULT CALLBACK WndProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);  

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int iCmdShow)
{
    WNDCLASS wc;
    HWND hwnd;
    HDC hDC;
    HGLRC hRC;        
    MSG msg;
    BOOL bQuit = FALSE;
    float theta = 0.0f;

    /* Register window class */
    wc.style = CS_OWNDC;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = "Xtreme3D";
    RegisterClass (&wc);
    
    /* Screen options */
    int s_real_width = GetSystemMetrics(SM_CXSCREEN);
    int s_real_height = GetSystemMetrics(SM_CYSCREEN);
    int s_width = 640;
    int s_height = 480;
    int s_depth = 32;
    BOOL s_fullscreen = false;

    /* Create main window */
    if (s_fullscreen) 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Demo", 
      WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
      SetWindowPos(hwnd,HWND_TOP,0,0,s_width,s_height,SWP_NOSIZE);
    }
    else 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Demo", 
      WS_CAPTION | WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
    }

    DEVMODE sDevMode;
    ZeroMemory(&sDevMode, sizeof(DEVMODE));
    sDevMode.dmSize = sizeof(DEVMODE);
    sDevMode.dmPelsWidth = s_width;
    sDevMode.dmFields |= DM_PELSWIDTH;
    sDevMode.dmPelsHeight = s_height;
    sDevMode.dmFields |= DM_PELSHEIGHT;
    sDevMode.dmBitsPerPel = s_depth;
    sDevMode.dmFields |= DM_BITSPERPEL;
    
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);
    
    /* Xtreme3D functions */    
    EngineCreate(double(int(hwnd)));        
    MouseShowCursor(0);    
    
    double matlib = MaterialLibraryCreate();
    MaterialLibraryActivate(matlib); 
    
    double view1 = ViewerCreate(0,0,s_width,s_height);  
    ViewerSetLighting(view1,1);
    ViewerSetBackgroundColor(view1,c_white);
    ViewerEnableFog(view1,1);
    ViewerSetFogColor(view1,c_white);
    ViewerSetFogDistance(view1,100,500);
    ViewerEnableVSync(view1,vsmNoSync);
    
    //RGB(0,0,255)
    
    double back = DummycubeCreate(0);
    double scene = DummycubeCreate(0); 
    double front = DummycubeCreate(0);
    
    double light = LightCreate(lsOmni,0);
    ObjectSetPosition(light,5,40,5);
    
    MaterialCreate("mGround","media/ground.jpg",0);
        
    double shadows = DummycubeCreate(scene);
    double sp = ShadowplaneCreate(64,64, 8,8, shadows, light, c_black, 0.6, scene);
    ObjectSetMaterial(sp,"mGround");
    ObjectPitch(sp,90);
    
    double cube = CubeCreate(5,5,5,shadows);
    ObjectSetPosition(cube,0,2.5,-5);
       
    double camPos = DummycubeCreate(scene);
    ObjectSetPosition(camPos,0,15,40);
    
    double camera = CameraCreate(camPos);
    CameraSetViewDepth(camera,500);
    CameraSetFocal(camera,120);
    CameraSetSceneScale(camera,0.5);
    ViewerSetCamera(view1,camera); 
    
    double font = WindowsBitmapfontCreate("Arial",10);
    double text_fps = HUDTextCreate(font,"FPS: ",front);
    TextSetColor(text_fps,c_black,0.6);
    ObjectSetPosition(text_fps,16,8,0);
    
    double deltax = 0;
    double deltay = 1;
    double pit = 0;
    double mx = 320;
    double my = 240;
    
    MouseSetPosition(mx,my);
   
    static char FPS[256];

    /* Program main loop */
    while (!bQuit)
    {
        /* Check for messages */
        if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        {
            if (msg.message == WM_QUIT)
            {
                bQuit = TRUE;
            }
            else
            {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
            }
        }
        else
        {
            /* Xtreme3D functions */
            if (GetKeyState(VK_SPACE) & 0x80) ObjectRotate(cube,2,2,0);
            if (GetKeyState('W')      & 0x80) ObjectMove(camPos,  -0.5);
            if (GetKeyState('S')      & 0x80) ObjectMove(camPos,   0.5);
            if (GetKeyState('A')      & 0x80) ObjectStrafe(camPos, 0.5);
            if (GetKeyState('D')      & 0x80) ObjectStrafe(camPos,-0.5);
            
            deltax = (mx-MouseGetX())/6;
            deltay = (my-MouseGetY())/6;
            ObjectRotate(camPos,0,deltax,0);
            pit = ObjectGetPitch(camera)+deltay;
            if (pit<89 || pit>-89) ObjectRotate(camera,deltay,0,0);
            MouseSetPosition(mx,my);
            
            Update();
            ViewerRender(view1);
            
            sprintf(FPS, "FPS = %.f", GetFPS() );
            TextSetText(text_fps, FPS);
            Sleep(2);
        }
    }

    EngineDestroy();
    FreeLibrary(hLibXtreme3D);
    
    DestroyWindow (hwnd);
    
    sDevMode.dmPelsWidth = s_real_width;
    sDevMode.dmPelsHeight = s_real_height;
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);

    return msg.wParam;
}


LRESULT CALLBACK WndProc (HWND hWnd, UINT message,
                          WPARAM wParam, LPARAM lParam)
{

    switch (message)
    {
    case WM_CREATE:
        return 0;
        
    case WM_CLOSE:
        PostQuitMessage (0);
        return 0;

    case WM_DESTROY:
        return 0;

    case WM_KEYDOWN:
        switch (wParam)
        {
        case VK_ESCAPE:
            {
            PostQuitMessage(0);
            return 0;
            }
        }
        return 0;

    default:
        return DefWindowProc (hWnd, message, wParam, lParam);
    }
}

/*
  VK_LBUTTON = 1;
  VK_RBUTTON = 2;
  VK_CANCEL = 3;
  VK_MBUTTON = 4;  { NOT contiguous with L & RBUTTON }
  VK_BACK = 8;
  VK_TAB = 9;
  VK_CLEAR = 12;
  VK_RETURN = 13;
  VK_SHIFT = $10;
  VK_CONTROL = 17;
  VK_MENU = 18;
  VK_PAUSE = 19;
  VK_CAPITAL = 20;
  VK_ESCAPE = 27;
  VK_SPACE = $20;
  VK_PRIOR = 33;
  VK_NEXT = 34;
  VK_END = 35;
  VK_HOME = 36;
  VK_LEFT = 37;
  VK_UP = 38;
  VK_RIGHT = 39;
  VK_DOWN = 40;
  VK_SELECT = 41;
  VK_PRINT = 42;
  VK_EXECUTE = 43;
  VK_SNAPSHOT = 44;
  VK_INSERT = 45;
  VK_DELETE = 46;
  VK_HELP = 47;
  VK_LWIN = 91;
  VK_RWIN = 92;
  VK_APPS = 93;
  VK_NUMPAD0 = 96;
  VK_NUMPAD1 = 97;
  VK_NUMPAD2 = 98;
  VK_NUMPAD3 = 99;
  VK_NUMPAD4 = 100;
  VK_NUMPAD5 = 101;
  VK_NUMPAD6 = 102;
  VK_NUMPAD7 = 103;
  VK_NUMPAD8 = 104;
  VK_NUMPAD9 = 105;
  VK_MULTIPLY = 106;
  VK_ADD = 107;
  VK_SEPARATOR = 108;
  VK_SUBTRACT = 109;
  VK_DECIMAL = 110;
  VK_DIVIDE = 111;
  VK_F1 = 112;
  VK_F2 = 113;
  VK_F3 = 114;
  VK_F4 = 115;
  VK_F5 = 116;
  VK_F6 = 117;
  VK_F7 = 118;
  VK_F8 = 119;
  VK_F9 = 120;
  VK_F10 = 121;
  VK_F11 = 122;
  VK_F12 = 123;
  VK_F13 = 124;
  VK_F14 = 125;
  VK_F15 = 126;
  VK_F16 = 127;
  VK_F17 = 128;
  VK_F18 = 129;
  VK_F19 = 130;
  VK_F20 = 131;
  VK_F21 = 132;
  VK_F22 = 133;
  VK_F23 = 134;
  VK_F24 = 135;
  VK_NUMLOCK = 144;
  VK_SCROLL = 145;
  VK_LSHIFT = 160;
  VK_RSHIFT = 161;
  VK_LCONTROL = 162;
  VK_RCONTROL = 163;
  VK_LMENU = 164;
  VK_RMENU = 165;
  VK_PROCESSKEY = 229;
  VK_ATTN = 246;
  VK_CRSEL = 247;
  VK_EXSEL = 248;
  VK_EREOF = 249;
  VK_PLAY = 250;
  VK_ZOOM = 251;
  VK_NONAME = 252;
  VK_PA1 = 253;
  VK_OEM_CLEAR = 254;
  */
