#include <stdio.h>
#include <windows.h>
#include <math.h>
#include "include/xtreme3d.h"
#include "include/engine.h"
#include "include/viewer.h"
#include "include/dummycube.h"
#include "include/camera.h"
#include "include/light.h"
#include "include/fonttext.h"
#include "include/primitives.h"
#include "include/object.h"
#include "include/material.h"
#include "include/mouse.h"
#include "include/actor.h"
#include "include/skybox.h"
#include "include/dce.h"

LRESULT CALLBACK WndProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam); 

static double CreateDceBox(double width, double height, double depth, double x, double y, double z, PCHAR material, double turn, double dce, double parent)
{ 
   static double col;
   col = CubeCreate(width,height,depth, parent);
   ObjectSetPosition(col,x,y,z);
   ObjectTurn(col,turn);
   DceStaticSetManager(col,dce);
   DceStaticSetActive(col,1);
   DceStaticSetSize(col,width,height,depth);  
   DceStaticSetShape(col,1); 
   DceStaticSetSolid(col,1);
   ObjectSetMaterial(col,material);
   return col; 
}

static double CreateSceneBox(double width, double height, double depth, double x, double y, double z, PCHAR material, double turn, double parent)
{ 
  static double col;
  col = CubeCreate(width,height,depth,parent);
  ObjectSetPosition(col,x,y,z);
  ObjectTurn(col,turn);
  ObjectSetMaterial(col,material);
  return col;
}

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine,
                    int iCmdShow)
{
    WNDCLASS wc;
    HWND hwnd;
    HDC hDC;
    HGLRC hRC;        
    MSG msg;
    BOOL bQuit = FALSE;
    float theta = 0.0f;

    /* Register window class */
    wc.style = CS_OWNDC;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) GetStockObject (BLACK_BRUSH);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = "Xtreme3D";
    RegisterClass (&wc);
    
    /* Screen options */
    int s_real_width = GetSystemMetrics(SM_CXSCREEN);
    int s_real_height = GetSystemMetrics(SM_CYSCREEN);
    int s_width = 640;
    int s_height = 480;
    int s_depth = 32;
    BOOL s_fullscreen = false;

    /* Create main window */
    if (s_fullscreen) 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Demo", 
      WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
      SetWindowPos(hwnd,HWND_TOP,0,0,s_width,s_height,SWP_NOSIZE);
    }
    else 
    {
      hwnd = CreateWindow (
      "Xtreme3D", "Xtreme3D Demo", 
      WS_CAPTION | WS_POPUPWINDOW | WS_VISIBLE,
      160, 120, s_width, s_height,
      NULL, NULL, hInstance, NULL);
    }

    DEVMODE sDevMode;
    ZeroMemory(&sDevMode, sizeof(DEVMODE));
    sDevMode.dmSize = sizeof(DEVMODE);
    sDevMode.dmPelsWidth = s_width;
    sDevMode.dmFields |= DM_PELSWIDTH;
    sDevMode.dmPelsHeight = s_height;
    sDevMode.dmFields |= DM_PELSHEIGHT;
    sDevMode.dmBitsPerPel = s_depth;
    sDevMode.dmFields |= DM_BITSPERPEL;
    
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);
    
    /* Xtreme3D functions */    
    EngineCreate(double(int(hwnd)));        
    MouseShowCursor(0);    
    
    double matlib = MaterialLibraryCreate();
    MaterialLibraryActivate(matlib); 
    
    double view1 = ViewerCreate(0,0,s_width,s_height);    
    ViewerEnableVSync(view1,1);
    ViewerSetLighting(view1,1);
    ViewerSetBackgroundColor(view1,c_black);
    ViewerEnableFog(view1,1);
    ViewerSetFogColor(view1,c_black);
    ViewerSetFogDistance(view1,100,500);
    
    //RGB(0,0,255)
    
    double back = DummycubeCreate(0);
    double scene = DummycubeCreate(0); 
    double front = DummycubeCreate(0);
    
    double light = LightCreate(lsParallel,0);
    LightSetDiffuseColor(light,RGB(255,128,0));
    LightSetAmbientColor(light,RGB(0,128,255));
    ObjectSetDirection(light,-1,-1,0);
    double light2 = LightCreate(lsParallel,0);
    LightSetDiffuseColor(light2,RGB(0,128,255));
    LightSetAmbientColor(light2,RGB(255,128,0));
    ObjectSetDirection(light2,1,-1,0);
    
    MaterialCreate("skyTop","media/sky_top.jpg",0);
    MaterialCreate("skyBottom","media/sky_bottom.jpg",0);
    MaterialCreate("skyLeft","media/sky_left.jpg",0);
    MaterialCreate("skyRight","media/sky_right.jpg",0);
    MaterialCreate("skyFront","media/sky_front.jpg",0);
    MaterialCreate("skyBack","media/sky_back.jpg",0);
    double sky = SkyboxCreate(back);
    SkyboxSetMaterial(sky,0,"skyTop");
    SkyboxSetMaterial(sky,1,"skyBottom");
    SkyboxSetMaterial(sky,2,"skyLeft");
    SkyboxSetMaterial(sky,3,"skyRight");
    SkyboxSetMaterial(sky,4,"skyFront");
    SkyboxSetMaterial(sky,5,"skyBack");
        
    double plane = PlaneCreate(0,100,100, 8,8, scene);
    MaterialCreate("plane_tex","media/ground.jpg",0);
    ObjectSetMaterial(plane,"plane_tex");
    ObjectSetPosition(plane,0,0.5,0);
    ObjectSetDirection(plane,0,1,0);
    
    double dce = DceManagerCreate();
    DceManagerSetManualStep(dce,1);
    DceManagerSetMovementScale(dce,2);
    double level = DummycubeCreate(scene);
    MaterialCreate("mbbox","",0);
    MaterialSetOptions("mbbox",0,1);
    MaterialSetPolygonMode("mbbox",pmLines);
    MaterialSetBlendingMode("mbbox",bmTransparency);
    MaterialSetDiffuseColor("mbbox",c_white,0);
    
    MaterialCreate("mbricks","media/wall.jpg",0);
    MaterialSetTextureScale("mbricks",5,1);
    MaterialCreate("mbricks2","media/wall2.jpg",0);
    MaterialCreate("mcolumn","media/column.jpg",0);
    MaterialSetTextureScale("mcolumn",1,10);

    double plane_col = CreateDceBox(100,100,100, 0,-50,0, "mbricks2", 0, dce, level);

    double blo1_col = CreateDceBox(100,10,10,     0,0,-55, "mbricks", 0, dce, level);
    double blo2_col = CreateDceBox(100,10,10,     0,0, 55, "mbricks", 0, dce, level);
    double blo3_col = CreateDceBox(100,10,10,   -55,0,  0, "mbricks", 90, dce, level);
    double blo4_col = CreateDceBox(100,10,10,    55,0,  0, "mbricks", 90, dce, level);

    double blo5 = CreateSceneBox(100,10,10,     0,95,-55, "mbricks", 0, level);
    double blo6 = CreateSceneBox(100,10,10,     0,95, 55, "mbricks", 0, level);
    double blo7 = CreateSceneBox(100,10,10,   -55,95,  0, "mbricks", 90, level);
    double blo8 = CreateSceneBox(100,10,10,    55,95,  0, "mbricks", 90, level);

    double colu1_col = CreateDceBox(10,200,10,    -55,0,-55, "mcolumn", 0, dce, level);
    double colu2_col = CreateDceBox(10,200,10,    -55,0, 55, "mcolumn", 0, dce, level);
    double colu3_col = CreateDceBox(10,200,10,     55,0, 55, "mcolumn", 0, dce, level);
    double colu4_col = CreateDceBox(10,200,10,     55,0,-55, "mcolumn", 0, dce, level);
    
    double player_col = SphereCreate(5,8,8,scene);
    ObjectSetMaterial(player_col,"mbbox");
    DceSetManager(player_col,dce);
    DceSetUseGravity(player_col,0);
    DceSetActive(player_col,1);
    DceSetSize(player_col,5,5,5);
    DceSetSolid(player_col,1);
    DceSetSlideOrBounce(player_col,0);
    ObjectSetPosition(player_col,0,5,0);
    DceSetFriction(player_col,1.5); 
    double grav = 0;
    double jump = 0;
    double floory = ObjectGetGroundHeight(player_col,level) + 5;
    ObjectRotate(player_col,0,90,0);
    
    double player_dummy = DummycubeCreate(scene);
    MaterialCreate("mPlayer","media/quake2model/player_tex.jpg",0);
    MaterialCreate("mWeapon","media/quake2model/weapon_tex.jpg",0);
    double actor = ActorCreate("media/quake2model/player_geom.md2",player_dummy);
    ObjectSetMaterial(actor,"mPlayer");
    ObjectSetScale(actor,0.2,0.2,0.2);
    ObjectRotate(actor,90,0,0);
    ObjectSetPosition(actor,0,-0.25,0);
    ActorSwitchToAnimation(actor,0);
    ActorSetInterval(actor, 80);
    double weapon = ActorCreate("media/quake2model/weapon_geom.md2",player_dummy);
    MaterialSetTextureCompression("mWeapon", tcHighQuality);
    ObjectSetMaterial(weapon,"mWeapon");
    ObjectSetScale(weapon,0.2,0.2,0.2);
    ActorSwitchToAnimation(weapon,0);
    PCHAR state = "idle";
    double current_frame;
    
    double camPos = DummycubeCreate(scene);
    double camera = CameraCreate(camPos);
    CameraSetViewDepth(camera,1000);
    CameraSetFocal(camera,120);
    CameraSetSceneScale(camera,0.5);
    ObjectSetPosition(camera,0,10,15);
    ViewerSetCamera(view1,camera);
    CameraSetTargetObject(camera,player_col);
    
    double font = WindowsBitmapfontCreate("Arial",10);
    double text_fps = HUDTextCreate(font,"FPS: ",front);
    TextSetColor(text_fps,c_white,0.6);
    ObjectSetPosition(text_fps,16,8,0);
    
    double deltax = 0;
    double deltay = 1;
    double pit = 0;
    double mx = 320;
    double my = 240;
    
    MouseSetPosition(mx,my);
   
    static char FPS[256];

    /* Program main loop */
    while (!bQuit)
    {
        /* Check for messages */
        if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        {
            if (msg.message == WM_QUIT)
            {
                bQuit = TRUE;
            }
            else
            {
                TranslateMessage (&msg);
                DispatchMessage (&msg);
            }
        }
        else
        {
            /* Xtreme3D functions */
            state = "idle";
            if      (GetKeyState('W') & 0x80) {if (state!="land") {DceApplyAcceleration(player_col, 10,0,0); if (state!="jump") state="run";}}
            else if (GetKeyState('S') & 0x80) {if (state!="land") {DceApplyAcceleration(player_col,-10,0,0); if (state!="jump") state="run";}}  
            if      (GetKeyState(VK_SPACE) & 0x80) {if (state!="land") jump=1.3;}
            
            DceMove(player_col,0,jump-grav,0, 0.5);
            floory = ObjectGetGroundHeight(player_col,level)+5;
            if (floor(ObjectGetPosition(player_col,1)) > floory)
            {
            state="jump";
            if (jump>0) jump-=0.05;
            grav+=0.05;
            }
            else
            {
            jump=0;
            grav=0;
            }
                     
            if (state=="idle") {ActorSetInterval(actor, 100); ActorSetAnimationRange(actor,0 ,39); ActorSetAnimationMode(actor,aamLoop);}
            if (state=="run")  {ActorSetInterval(actor, 80); ActorSetAnimationRange(actor,40,45); ActorSetAnimationMode(actor,aamLoop);}
            if (state=="jump") {ActorSetInterval(actor, 200); ActorSetAnimationRange(actor,66,73); ActorSetAnimationMode(actor,aamPlayOnce);}
            
            ObjectAlignWithObject(player_dummy,player_col);
            ObjectSetPositionOfObject(player_dummy,player_col);
            ObjectAlignWithObject(weapon,actor);
            ObjectSetPositionOfObject(weapon,actor);
            ActorSynchronize(weapon,actor);
            
            double cx=ObjectGetAbsolutePosition(camPos,0);
            double cy=ObjectGetAbsolutePosition(camPos,1);
            double cz=ObjectGetAbsolutePosition(camPos,2);
            double tx=ObjectGetAbsolutePosition(player_col,0);
            double ty=ObjectGetAbsolutePosition(player_col,1);
            double tz=ObjectGetAbsolutePosition(player_col,2);
            double dx=tx-cx; 
            double dy=ty+2-cy; 
            double dz=tz-cz;
            ObjectTranslate(camPos,dx,dy*0.05,dz);
            
            deltax = (mx-MouseGetX())/6;
            deltay = (my-MouseGetY())/6;
            CameraMoveAroundTarget(camera,0,deltax);
            ObjectRotate(player_col,0,deltax,0);
            CameraMoveAroundTarget(camera,deltay,0);
            MouseSetPosition(mx,my);
            
            DceManagerStep(dce,0.1);
            Update();
            ViewerRender(view1);
            
            sprintf(FPS, "FPS = %.f", GetFPS() );
            TextSetText(text_fps, FPS);
            Sleep(1);
        }
    }

    EngineDestroy();
    FreeLibrary(hLibXtreme3D);
    
    DestroyWindow (hwnd);
    
    sDevMode.dmPelsWidth = s_real_width;
    sDevMode.dmPelsHeight = s_real_height;
    if (s_fullscreen) ChangeDisplaySettings(&sDevMode, CDS_UPDATEREGISTRY);

    return msg.wParam;
}


LRESULT CALLBACK WndProc (HWND hWnd, UINT message,
                          WPARAM wParam, LPARAM lParam)
{

    switch (message)
    {
    case WM_CREATE:
        return 0;
        
    case WM_CLOSE:
        PostQuitMessage (0);
        return 0;

    case WM_DESTROY:
        return 0;

    case WM_KEYDOWN:
        switch (wParam)
        {
        case VK_ESCAPE:
            {
            PostQuitMessage(0);
            return 0;
            }
        }
        return 0;

    default:
        return DefWindowProc (hWnd, message, wParam, lParam);
    }
}
