/*
GLBaseObject
Basic scene graph object class.
*/
class GLBaseObject
{
public:
  GLBaseObject() { }
  virtual ~GLBaseObject() { Destroy(); }
  void Release() { delete this; }
  virtual void Update()
  {
    for( std::list<GLBaseObject*>::iterator i = m_lstChildren.begin();
         i != m_lstChildren.end(); i++ )
    {
      (*i)->Update();
    }
  }
  void Destroy()
  {
    for( std::list<GLBaseObject*>::iterator i = m_lstChildren.begin();
         i != m_lstChildren.end(); i++ )
      (*i)->Release();
  
    m_lstChildren.clear();
  }
  void AddChild( GLBaseObject* pNode )
  {
    m_lstChildren.push_back(pNode);
  }
protected:
  std::list<GLBaseObject*> m_lstChildren;
};

/*
GLBaseSceneObject
Basic 3D object class.
*/
class GLBaseSceneObject: public GLBaseObject
{
public:  
  GLVector3f Scaling;
  GLVector4f Position;
  GLVector3f Rotation;
  GLColor Color;
  bool UseObjectColor;
  bool Visible;
  GLMaterial Material;
  GLMaterial Material2;
  float CustomParameter1;
  float CustomParameter2;
  float CustomParameter3;
  float CustomParameter4;
  GLBaseSceneObject()
  {
  	Color.R=1;
  	Color.G=1;
  	Color.B=1;
  	Color.A=1;
  	UseObjectColor=false;
  	Visible=true;
	Scaling.X = 1;
        Scaling.Y = 1;
        Scaling.Z = 1;
        Position.X = 0;
        Position.Y = 0;
	Position.Z = 0;
	Position.W = 0;
	Rotation.X = 0;
	Rotation.Y = 0;
	Rotation.Z = 0;
	CustomParameter1 = 0;
	CustomParameter2 = 0;
	CustomParameter3 = 0;
	CustomParameter4 = 0;
  }
  ~GLBaseSceneObject() { }
  void SetColor(float r, float g, float b, float a)
  {
  	Color.R=r;
  	Color.G=g;
  	Color.B=b;
  	Color.A=a;
  }
  void Rotate(float p, float t, float r)
  {
	Rotation.X+=p;
	Rotation.Y+=t;
	Rotation.Z+=r;
  }
  void SetRotation(float p, float t, float r)
  {
	Rotation.X=p;
	Rotation.Y=t;
	Rotation.Z=r;
  }
  void SetPosition(float x, float y, float z)
  {
	Position.X = x;
	Position.Y = y;
	Position.Z = z;
  }
  void Translate(float x, float y, float z)
  {
	Position.X += x;
	Position.Y += y;
	Position.Z += z;
  }
  void Scale(float x, float y, float z)
  {
	Scaling.X+=x;
	Scaling.Y+=y;
	Scaling.Z+=z;
  }
  void SetScaling(float x, float y, float z)
  {
	Scaling.X=x;
	Scaling.Y=y;
	Scaling.Z=z;
  }
  void FreeFly(float spd)
  {
  	float trad, prad;
  	prad = (Rotation.X / 180 * 3.141592654f);
  	trad = (Rotation.Y / 180 * 3.141592654f);
  	Position.X -= float(sin(trad))*spd;
  	Position.Y += float(sin(prad))*spd;
  	Position.Z -= float(cos(trad))*spd;
  }
  void Move(float spd)
  {
  	float trad, prad;
  	prad = (Rotation.X / 180 * 3.141592654f);
  	trad = (Rotation.Y / 180 * 3.141592654f);
  	Position.X -= float(sin(trad))*spd;
  	Position.Z -= float(cos(trad))*spd;
  }
  void Strafe(float spd)
  {
  	float trad, prad;
  	prad = (Rotation.X / 180 * 3.141592654f);
  	trad = (Rotation.Y / 180 * 3.141592654f);
  	Position.X += float(cos(trad))*spd;
  	Position.Z -= float(sin(trad))*spd;
  }
  void Update()
  {
  	glTranslatef (Position.X, Position.Y, Position.Z);
  	glRotatef(Rotation.X, 1.0f, 0.0f, 0.0f);
  	glRotatef(Rotation.Y, 0.0f, 1.0f, 0.0f);
  	glRotatef(Rotation.Z, 0.0f, 0.0f, 1.0f);
        glScalef(Scaling.X,Scaling.Y,Scaling.Z);
  	glColor4f(1,1,1,1);
  	GLBaseObject::Update();
  	Material.Update();
  	if (UseObjectColor) 
  	{
  		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  		glColor4f(Color.R,Color.G,Color.B,Color.A);
  	}
  }
};

/*
GLCamera
Camera object class.
*/
class GLCamera: public GLBaseSceneObject
{
public:
  GLCamera() { }
  ~GLCamera() { }
  void Update()
  { 
  glPushMatrix ();
  glTranslatef (Position.X, Position.Y, Position.Z);
  glRotatef(Rotation.Y,  0.0f, 1.0f, 0.0f);
  glRotatef(Rotation.X, 1.0f, 0.0f, 0.0f);
  glRotatef(Rotation.Z,  0.0f, 0.0f, 1.0f);
  glScalef(Scaling.X,Scaling.Y,Scaling.Z);
  GLBaseObject::Update();
  glPopMatrix ();
  }
};

/*
GLLight
Light source object class.
*/
class GLLight: public GLBaseSceneObject
{
public:
  GLfloat Ambient[4];
  GLfloat Diffuse[4];
  GLfloat Specular[4];
  GLfloat LightPosition[4];
  GLenum Light;
  bool Shining;
  void SetDiffuseColor(float r, float g, float b, float a)
  {
  Diffuse[0]=r;
  Diffuse[1]=g;
  Diffuse[2]=b;
  Diffuse[3]=a;
  }
  void SetAmbientColor(float r, float g, float b, float a)
  {
  Ambient[0]=r;
  Ambient[1]=g;
  Ambient[2]=b;
  Ambient[3]=a;
  }
  void SetSpecularColor(float r, float g, float b, float a)
  {
  Specular[0]=r;
  Specular[1]=g;
  Specular[2]=b;
  Specular[3]=a;
  }
  GLLight() 
  {
  Ambient[0]=0;
  Ambient[1]=0;
  Ambient[2]=0;
  Ambient[3]=1;
  Diffuse[0]=1;
  Diffuse[1]=1;
  Diffuse[2]=1;
  Diffuse[3]=1;
  Specular[0]=1;
  Specular[1]=1;
  Specular[2]=1;
  Specular[3]=1;
  LightPosition[0]=Position.X;
  LightPosition[1]=Position.Y;
  LightPosition[2]=Position.Z;
  LightPosition[3]=Position.W;
  Light=GL_LIGHT0;
  Shining=true;
  }
  ~GLLight() { }
  void Update()
  { 
    glPushMatrix ();
    GLBaseSceneObject::Update();
    if (Visible) Render();
    Material.PostUpdate();
    glPopMatrix ();
  }
  void Render()
  {
    LightPosition[0]=Position.X;
    LightPosition[1]=Position.Y;
    LightPosition[2]=Position.Z;
    LightPosition[3]=Position.W;
    glEnable(Light);
    glLightfv(Light, GL_AMBIENT,  Ambient);
    glLightfv(Light, GL_DIFFUSE,  Diffuse);
    glLightfv(Light, GL_SPECULAR, Specular);
    glLightfv(Light, GL_POSITION, LightPosition);
  }
};

/*
GLDummy
Empty object class (used for storing a transformation).
*/
class GLDummy: public GLBaseSceneObject
{
public:
  GLDummy() 
  { 
  Visible=false; 
  }
  ~GLDummy() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  glColor4f(1,1,1,1);
  glDisable(GL_LIGHTING);
  glutWireCube(1);
  glEnable(GL_LIGHTING);
  }
};

/*
GLCube
Cube primitive.
*/
class GLCube: public GLBaseSceneObject
{
public:
  float Size;
  bool Solid;
  GLCube() 
  { 
  Size=1;
  Solid=true; 
  }
  ~GLCube() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  glutSolidCube(Size);
  }
};

/*
GLSphere
Sphere primitive.
*/
class GLSphere: public GLBaseSceneObject
{
public:
  float Size;
  int Slices;
  int Stacks;
  GLUquadricObj * Quadric;
  GLSphere() 
  { 
  Size=1;
  Slices=32;
  Stacks=32;
  Quadric = gluNewQuadric();
  gluQuadricDrawStyle(Quadric, GLU_FILL);
  gluQuadricNormals(Quadric, GLU_SMOOTH);
  gluQuadricTexture(Quadric,GL_TRUE);
  }
  ~GLSphere() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  gluSphere(Quadric, Size, Slices, Stacks);
  }
};

/*
GLCylinder
Cylinder primitive.
*/
class GLCylinder: public GLBaseSceneObject
{
public:
  int Slices;
  int Stacks;
  int Loops;
  float Base;
  float Top;
  float Height;
  GLUquadricObj * Quadric;
  GLCylinder() 
  { 
  Slices=32;
  Stacks=32;
  Loops=3;
  Base=0.5;
  Top=0.5;
  Height=1;
  Quadric = gluNewQuadric();
  gluQuadricDrawStyle(Quadric, GLU_FILL);
  gluQuadricNormals(Quadric, GLU_SMOOTH);
  gluQuadricTexture(Quadric,GL_TRUE);
  }
  ~GLCylinder() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  glPushMatrix ();     
  glTranslatef(0,0,-Height/2);    
  gluCylinder(Quadric, Base, Top, Height, Slices, Stacks);
  Material2.Update();
  glTranslatef(0,0,Height);       
  gluPartialDisk (Quadric, 0, Top, Slices, Loops, 0, 360);
  glTranslatef(0,0,-Height);
  glRotatef(180,0,1,0);       
  gluPartialDisk (Quadric, 0, Base, Slices, Loops, 0, 360);
  glPopMatrix ();
  Material2.PostUpdate();
  }
};

/*
GLGeometry
3D geometry class template.
*/
class GLGeometry: public GLBaseSceneObject
{
public:
  GLGeometry() { }
  ~GLGeometry() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();

/*
glPolygonMode(GL_FRONT, GL_LINE);
glPolygonMode(GL_BACK, GL_LINE);
glCullFace(GL_FRONT);
glLineWidth(2.0f);
glEnable(GL_LINE_SMOOTH);
if (Visible) Render();
glPolygonMode(GL_FRONT, GL_FILL);
glPolygonMode(GL_BACK, GL_FILL);
glCullFace(GL_BACK);
*/

  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  glBegin (GL_QUADS);
     glTexCoord2f(1.0f, 0.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f (-1.0f, 0.0f,-1.0f);
     glTexCoord2f(1.0f, 1.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f (-1.0f, 0.0f, 1.0f);
     glTexCoord2f(0.0f, 1.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f ( 1.0f, 0.0f, 1.0f);
     glTexCoord2f(0.0f, 0.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f ( 1.0f, 0.0f,-1.0f);
  glEnd ();
  }
};
    
/*
GLPlane
Plane primitive.
*/
class GLPlane: public GLBaseSceneObject
{
public:
  int TilesX;
  int TilesY;
  GLPlane() 
  {
  	TilesX = 5;
	TilesY = 5; 
  }
  ~GLPlane() { }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  //for( int i=0; i<=TilesX; i++ ) 
//	{
//	for( int i=0; i<=TilesY; i++ ) 
//		{
		glBegin (GL_QUADS);
     		glTexCoord2f(1.0f, 0.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f(-1.0f, 0.0f,-1.0f);
     		glTexCoord2f(1.0f, 1.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f(-1.0f, 0.0f, 1.0f);
     		glTexCoord2f(0.0f, 1.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f( 1.0f, 0.0f, 1.0f);
     		glTexCoord2f(0.0f, 0.0f); glNormal3f(0.0f, 1.0f, 0.0f); glVertex3f( 1.0f, 0.0f,-1.0f);
  		glEnd ();
//		}
//	}
  }
};   

//GLHudSprite
class GLHudSprite: public GLBaseSceneObject
{
public:
  int DrawPosition;
  GLHudSprite() 
  {
    DrawPosition=0; 
  }
  ~GLHudSprite() { }
  void Update()
  { 

  //glViewport(0,0,512,512);  
  //glBindTexture(GL_TEXTURE_2D, Material.Texture.ID);
  //glCopyTexImage2D(GL_TEXTURE_2D,0,GL_RGB,0,0,512,512,0);
  //glViewport(0,0,640,480);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0, d_window_width, d_window_height, 0, -1, 1);
  glMatrixMode(GL_MODELVIEW);

  glLoadIdentity();

  glTranslatef(Position.X, Position.Y, Position.Z);
  glRotatef(Rotation.X, 1.0f, 0.0f, 0.0f);
  glRotatef(Rotation.Y, 0.0f, 1.0f, 0.0f);
  glRotatef(Rotation.Z, 0.0f, 0.0f, 1.0f);
  glScalef(Scaling.X,Scaling.Y,Scaling.Z);
  glColor4f(1,1,1,1);
  GLBaseObject::Update();
  //Material.Update();
  if (UseObjectColor) 
  {
  		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  		glColor4f(Color.R,Color.G,Color.B,Color.A);
  }

  glDisable(GL_CULL_FACE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_NORMALIZE);
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_LIGHTING);
    
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, Material.Texture.ID);
  glColor3f(1.0, 1.0, 1.0);
  
    glBegin(GL_QUADS);
    if (DrawPosition==0)
    {
    glTexCoord2f(0.0f, 1.0f); glVertex2f(0-0.5,0-0.5);
    glTexCoord2f(1.0f, 1.0f); glVertex2f(1-0.5,0-0.5);
    glTexCoord2f(1.0f, 0.0f); glVertex2f(1-0.5,1-0.5);
    glTexCoord2f(0.0f, 0.0f); glVertex2f(0-0.5,1-0.5);
    }
    if (DrawPosition==1)
    {
    glTexCoord2f(0.0f, 1.0f); glVertex2f(0,0);
    glTexCoord2f(1.0f, 1.0f); glVertex2f(1,0);
    glTexCoord2f(1.0f, 0.0f); glVertex2f(1,1);
    glTexCoord2f(0.0f, 0.0f); glVertex2f(0,1);
    }
    glEnd();
    
  glDisable(GL_TEXTURE_2D);
    
    glEnable(GL_CULL_FACE);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_NORMALIZE);
    glEnable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHTING);

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  }
};

/*
GLFileObject
*/
class GLFileObject: public GLBaseSceneObject
{
public:
  GLTriangle Polygon[128];
  int VCount;
  int PCount;
  GLFileObject() 
  {
  }
  ~GLFileObject() { }
  void Load(char *filename)
  {	
	FILE *file = fopen(filename, "r");
	if (file!=NULL)
		{
		PCount=(int)ReadNextFloatFromFile(&file);
		for (int i=1; i<=PCount; i++)
			{	    
			Polygon[i].Vertex1.Position.X=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Position.Y=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Position.Z=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Color.R=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Color.G=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Color.B=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex1.Color.A=ReadNextFloatFromFile(&file);
			
			Polygon[i].Vertex2.Position.X=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Position.Y=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Position.Z=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Color.R=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Color.G=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Color.B=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex2.Color.A=ReadNextFloatFromFile(&file);
			
			Polygon[i].Vertex3.Position.X=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Position.Y=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Position.Z=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Color.R=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Color.G=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Color.B=ReadNextFloatFromFile(&file);
			Polygon[i].Vertex3.Color.A=ReadNextFloatFromFile(&file);
			}
		}
	else 
		{
		printf("GLFileObject: Cannot open file \"%s\"", filename);
		exit(0);
		}

  }
  void Update()
  {
  glPushMatrix ();
  GLBaseSceneObject::Update();
  if (Visible) Render();
  Material.PostUpdate();
  glPopMatrix ();
  }
  void Render()
  {
  glDisable(GL_NORMALIZE);
  glDisable(GL_LIGHTING);
  glDisable(GL_CULL_FACE);
  	 for (int i=1; i<=PCount; i++)
		{
		glBegin (GL_TRIANGLES);
		//glTexCoord2f(1.0f, 0.0f); 
		//glNormal3f(0.0f, 1.0f, 0.0f); 
		glColor4f(Polygon[i].Vertex1.Color.R, Polygon[i].Vertex1.Color.G, Polygon[i].Vertex1.Color.B, Polygon[i].Vertex1.Color.A);
			glVertex3f (Polygon[i].Vertex1.Position.X, Polygon[i].Vertex1.Position.Y, Polygon[i].Vertex1.Position.Z);
		glColor4f(Polygon[i].Vertex2.Color.R, Polygon[i].Vertex2.Color.G, Polygon[i].Vertex2.Color.B, Polygon[i].Vertex2.Color.A);
			glVertex3f (Polygon[i].Vertex2.Position.X, Polygon[i].Vertex2.Position.Y, Polygon[i].Vertex2.Position.Z);
		glColor4f(Polygon[i].Vertex3.Color.R, Polygon[i].Vertex3.Color.G, Polygon[i].Vertex3.Color.B, Polygon[i].Vertex3.Color.A);	
			glVertex3f (Polygon[i].Vertex3.Position.X, Polygon[i].Vertex3.Position.Y, Polygon[i].Vertex3.Position.Z);
	    glEnd ();
		}
  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHTING);
  glEnable(GL_NORMALIZE);
  }
};
