#include "gmbRigidBodyExports.h"

#include "btBulletDynamicsCommon.h"
#include "gmbMotionState.h"
#include "gmbShape.h"
#include "GMFunctions.h"

gmbIndexedList<gmbRigidBody*> gmbGlobal_RigidBodies;

//Basic Body Functions 
GM_export GM_Real GMBULLET_AddRigidBody(GM_Real mass,GM_Real type,GM_Real shapeID,GM_Real x,GM_Real y,GM_Real z,GM_Real r1,GM_Real r2,GM_Real r3)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbShape* tShape;
	btCollisionShape* tcs=NULL;
	if(shapeID==-1)
		tcs=NULL;
	else if((tShape=gmbGlobal_ShapeList.GetInstance(unsigned int(shapeID))))
		tcs=tShape->getShape();
	else
		return -2;
			
	btTransform startTransform;
	startTransform.setIdentity();
	startTransform.setBasis(RotationWithFlags(btScalar(r1),btScalar(r2),btScalar(r3)));
	startTransform.setOrigin(btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);

	gmbMotionState* tMotionstate=new gmbMotionState();
	tMotionstate->setWorldTransform(startTransform);

	btVector3 tInertia(.0f,.0f,.0f);
	if(tcs && type==1)
		tShape->getShape()->calculateLocalInertia(btScalar(mass),tInertia);

	gmbRigidBody* tb=new gmbRigidBody(btScalar(mass),tcs,tMotionstate,tInertia,int(type));
	tMotionstate->Release();

	gmbGlobal_World->addRigidBody(tb);

	unsigned int tind=gmbGlobal_RigidBodies.AddInstance(tb);

	tb->m_GMBIndex=tind;
	tb->m_GMLIndex=0;
	return (GM_Real)tind;
}
GM_export GM_Real GMBULLET_AddRigidBodiesFromList(GM_Real mass,GM_Real type,GM_Real shapeListID,GM_Real bodyListID,GM_Real gmID,GM_Real x,GM_Real y,GM_Real z,GM_Real r1,GM_Real r2,GM_Real r3)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	btTransform startTransform;
	startTransform.setIdentity();
	startTransform.setBasis(RotationWithFlags(btScalar(r1),btScalar(r2),btScalar(r3)));
	startTransform.setOrigin(btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);

	for(int i=0;i<ds_list_size(shapeListID);i++)
	{
		unsigned int shapeID=unsigned int(ds_list_find_value(shapeListID,i));
		btCollisionShape* tcs=NULL;
		gmbShape* tShape;
		if(shapeID==-1)
			tcs=NULL;
		if(shapeID==-1)
			tcs=NULL;
		else if((tShape=gmbGlobal_ShapeList.GetInstance(unsigned int(shapeID))))
			tcs=tShape->getShape();
		else
			continue;
			

		gmbMotionState* tMotionstate=new gmbMotionState();
		tMotionstate->setWorldTransform(startTransform);

		btVector3 tInertia(.0f,.0f,.0f);
		if(tcs && type==1)
		{
			tShape->getShape()->calculateLocalInertia(btScalar(mass),tInertia);
		}
		gmbRigidBody* tb=new gmbRigidBody(btScalar(mass),tcs,tMotionstate,tInertia,int(type));
		tMotionstate->Release();
		if(tcs)
			gmbGlobal_World->addRigidBody(tb);
		unsigned int tind=gmbGlobal_RigidBodies.AddInstance(tb);
		tb->m_GMBIndex=tind;
		tb->m_GMLIndex=unsigned int(gmID);
		ds_list_add(bodyListID,tind);
	}
	return ds_list_size(bodyListID);
}
GM_export GM_Real GMBULLET_DeleteRigidBody(GM_Real ID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;

	btRigidBody* tb;
	if(!(tb=gmbGlobal_RigidBodies.GetInstance(int(ID))))
		return -1;

	for(int i=tb->getNumConstraintRefs() ; i>0 ; i--)
	{
		for(int j=0; j<gmbGlobal_Constraints.m_indexArray.size() ; j++)
		{
			if(gmbGlobal_Constraints.m_indexArray[j]->GetConstraint()==tb->getConstraintRef(i-1))
			{
				GMBULLET_DeleteConstraint(j);
				break;
			}
		}		
	}
	gmbGlobal_World->removeRigidBody(tb);
	gmbGlobal_RigidBodies.RemoveInstance(int(ID));
	return 0;
}
GM_export GM_Real GMBULLET_DeleteRigidBodiesFromStack(GM_Real stackID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	while(!ds_stack_empty(stackID))
	{
		unsigned int ID=unsigned int(ds_stack_pop(stackID));
		btRigidBody* tb;
		if(!(tb=gmbGlobal_RigidBodies.GetInstance(ID)))
		{
			continue;
		}

		while(tb->getNumConstraintRefs())
		{
			gmbConstraint* tc=dynamic_cast<gmbConstraint*>(tb->getConstraintRef(0));
			if(tc)
				GMBULLET_DeleteConstraint(gmbGlobal_Constraints.m_indexArray.findLinearSearch(tc));			
		}
		gmbGlobal_World->removeRigidBody(tb);
		gmbGlobal_RigidBodies.RemoveInstance(ID);
	}
	return 0;
}
GM_export GM_Real GMBULLET_DeleteRigidBodiesFromList(GM_Real listID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	for(int i=0;i<ds_list_size(listID);i++)
	{
		unsigned int ID=unsigned int(ds_list_find_value(listID,i));
		btRigidBody* tb;
		if(!(tb=gmbGlobal_RigidBodies.GetInstance(ID)))
		{
			continue;
		}

		while(tb->getNumConstraintRefs())
		{
			gmbConstraint* tc=dynamic_cast<gmbConstraint*>(tb->getConstraintRef(0));
			if(tc)
				GMBULLET_DeleteConstraint(gmbGlobal_Constraints.m_indexArray.findLinearSearch(tc));			
		}
		gmbGlobal_World->removeRigidBody(tb);
		gmbGlobal_RigidBodies.RemoveInstance(ID);
	}
	return 0;
}
GM_export GM_Real GMBULLET_SetFriction(GM_Real ID,GM_Real friction)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setFriction(btScalar(friction));
	return 0;
}
GM_export GM_Real GMBULLET_SetAllFriction(GM_Real friction)
{
	for(int i=0;i<gmbGlobal_World->getCollisionObjectArray().size();i++)
	{
		gmbGlobal_World->getCollisionObjectArray()[i]->setFriction(btScalar(friction));
	}
	return 0;
}
GM_export GM_Real GMBULLET_DisableDeactivation(GM_Real ID,GM_Real deactivate)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setActivationState(deactivate ? DISABLE_DEACTIVATION : ACTIVE_TAG);
	return 0;
}
GM_export GM_Real GMBULLET_ActivateBody(GM_Real ID,GM_Real forceActivation)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->activate(bool(forceActivation));
	return 0;
}
GM_export GM_Real GMBULLET_SetCollisionGroupMask(GM_Real ID,GM_Real cGroup,GM_Real cMask)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->getBroadphaseProxy()->m_collisionFilterGroup=(short)(int)cGroup;
	tb->getBroadphaseProxy()->m_collisionFilterMask=(short)(int)cMask;
	tb->activate();
	return 0;
}
GM_export GM_Real GMBULLET_AssociateBodyWithInstance(GM_Real ID,GM_Real Instance)
{	
	gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->m_GMLIndex=unsigned int(Instance);
	return 0;
}
GM_export GM_Real GMBULLET_SetMass(GM_Real ID,GM_Real mass)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv;
	tb->getCollisionShape()->calculateLocalInertia(btScalar(mass),tv);
	tb->setMassProps(btScalar(mass),tv);
	return 0;
}
GM_export GM_Real GMBULLET_SetDeactivationThresholds(GM_Real ID,GM_Real lin,GM_Real ang)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv;
	tb->setSleepingThresholds(btScalar(lin),btScalar(ang));
	return 0;
}
GM_export GM_Real GMBULLET_SetDeactivationTime(GM_Real ID,GM_Real time)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setDeactivationTime(btScalar(time));
	return 0;
}
GM_export GM_Real GMBULLET_SetHitFraction(GM_Real ID,GM_Real HitFraction)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv;
	tb->setHitFraction(btScalar(HitFraction));
	return 0;
}
GM_export GM_Real GMBULLET_SetDamping(GM_Real ID,GM_Real linDamping,GM_Real angDamping)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv;
	tb->setDamping(btScalar(linDamping),btScalar(angDamping));
	return 0;
}
GM_export GM_Real GMBULLET_SetRestitution(GM_Real ID,GM_Real rest)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv;
	tb->setRestitution(btScalar(rest));
	return 0;
}

//Kinetmatic Body Functions
GM_export GM_Real GMBULLET_SetPosition(GM_Real ID,GM_Real x, GM_Real y, GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setOrigin(btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetPositionX(GM_Real ID,GM_Real x)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setOrigin(tt.getOrigin()+btVector3(btScalar(x),0,0)*invDistanceScale);
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetPositionY(GM_Real ID,GM_Real y)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setOrigin(tt.getOrigin()+btVector3(0,btScalar(y),0)*invDistanceScale);
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetPositionZ(GM_Real ID,GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setOrigin(tt.getOrigin()+btVector3(0,0,btScalar(z))*invDistanceScale);
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetRelativePosition(GM_Real ID,GM_Real x, GM_Real y, GM_Real z)
{
	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setOrigin(tt.getOrigin()+btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_ClearRotation(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(btQuaternion(0,0,0));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_AddArbitraryRotation(GM_Real ID,GM_Real x,GM_Real y,GM_Real z,GM_Real angle)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt*=btTransform(btQuaternion(btVector3(btScalar(x),btScalar(y),btScalar(z)),btScalar(angle)*invAngleMultiplier));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_AddRotationWithFlags(GM_Real ID,GM_Real r1,GM_Real r2,GM_Real r3)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setBasis(RotationWithFlags(btScalar(r1),btScalar(r2),btScalar(r3),tt.getBasis()));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_AddRotationQuaternion(GM_Real ID,GM_Real x,GM_Real y,GM_Real z,GM_Real w)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(tt.getRotation()*btQuaternion(btScalar(x),btScalar(y),btScalar(z),btScalar(w)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_AddRotationSLERP(GM_Real ID,GM_Real x2,GM_Real y2,GM_Real z2,GM_Real w2,GM_Real amount)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(tt.getRotation().slerp(btQuaternion(btScalar(x2),btScalar(y2),btScalar(z2),btScalar(w2)),btScalar(amount)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_AddRotationSLERPEuler(GM_Real ID,GM_Real rx2,GM_Real ry2,GM_Real rz2,GM_Real amount)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(tt.getRotation().slerp(btQuaternion(btScalar(rx2),btScalar(ry2),btScalar(rz2)),btScalar(amount)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetArbitraryRotation(GM_Real ID,GM_Real x,GM_Real y,GM_Real z,GM_Real angle)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(btQuaternion(btVector3(btScalar(x),btScalar(y),btScalar(z)),btScalar(angle)*invAngleMultiplier));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetRotationWithFlags(GM_Real ID,GM_Real r1,GM_Real r2,GM_Real r3)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setBasis(RotationWithFlags(btScalar(r1),btScalar(r2),btScalar(r3)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetRotationQuaternion(GM_Real ID,GM_Real x,GM_Real y,GM_Real z,GM_Real w)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	tt.setRotation(btQuaternion(btScalar(x),btScalar(y),btScalar(z),btScalar(w)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetRotationSLERP(GM_Real ID,GM_Real x1,GM_Real y1,GM_Real z1,GM_Real w1,GM_Real x2,GM_Real y2,GM_Real z2,GM_Real w2,GM_Real amount)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	
	tt.setRotation(btQuaternion(btScalar(x1),btScalar(y1),btScalar(z1),btScalar(w1)).slerp(btQuaternion(btScalar(x2),btScalar(y2),btScalar(z2),btScalar(w2)),btScalar(amount)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetRotationSLERPEuler(GM_Real ID,GM_Real rx1,GM_Real ry1,GM_Real rz1,GM_Real rx2,GM_Real ry2,GM_Real rz2,GM_Real amount)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	
	tt.setRotation(btQuaternion(btScalar(rx1),btScalar(ry1),btScalar(rz1)).slerp(btQuaternion(btScalar(rx2),btScalar(ry2),btScalar(rz2)),btScalar(amount)));
	tb->getMotionState()->setWorldTransform(tt);
	return 0;
}
GM_export GM_Real GMBULLET_SetBodyPosFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(tt.getOrigin()+btVector3(tx,ty,tz)*invDistanceScale);
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(btVector3(tx,ty,tz)*invDistanceScale);
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyRotWFFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setBasis(RotationWithFlags(tx,ty,tz,tt.getBasis()));
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setBasis(RotationWithFlags(tx,ty,tz));
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyPosRotWFFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar trotz=float(ds_stack_pop(stackID));
			btScalar troty=float(ds_stack_pop(stackID));
			btScalar trotx=float(ds_stack_pop(stackID));
			btScalar tposz=float(ds_stack_pop(stackID));
			btScalar tposy=float(ds_stack_pop(stackID));
			btScalar tposx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(tt.getOrigin()+btVector3(tposx,tposy,tposz)*invDistanceScale);
				tt.setBasis(RotationWithFlags(trotx,troty,trotz,tt.getBasis()));
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar trotz=float(ds_stack_pop(stackID));
			btScalar troty=float(ds_stack_pop(stackID));
			btScalar trotx=float(ds_stack_pop(stackID));
			btScalar tposz=float(ds_stack_pop(stackID));
			btScalar tposy=float(ds_stack_pop(stackID));
			btScalar tposx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(btVector3(tposx,tposy,tposz)*invDistanceScale);
				tt.setBasis(RotationWithFlags(trotx,troty,trotz));
				tb->getMotionState()->setWorldTransform(tt);
			}
		}
	}
	return retval;
}

GM_export GM_Real GMBULLET_SetBodyPosFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(tt.getOrigin()+btVector3(btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyRotWFFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setBasis(RotationWithFlags(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++)),tt.getBasis()));
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setBasis(RotationWithFlags(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++))));
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyPosRotWFFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(tt.getOrigin()+btVector3(btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tt.setBasis(RotationWithFlags(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
					btScalar(ds_list_find_value(listID,i++)),tt.getBasis()));
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				btTransform tt;
				tb->getMotionState()->getWorldTransform(tt);
				tt.setOrigin(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tt.setBasis(RotationWithFlags(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++))));
				tb->getMotionState()->setWorldTransform(tt);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}

//Dynamic Body Functions
GM_export GM_Real GMBULLET_SetAngularFactor(GM_Real ID,GM_Real angFact)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setAngularFactor(btScalar(angFact));
	return 0;
}
GM_export GM_Real GMBULLET_SetAngularVelocityX(GM_Real ID,GM_Real x)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setX(btScalar(x)*invAngleMultiplier);
	tb->setAngularVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetAngularVelocityY(GM_Real ID,GM_Real y)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setY(btScalar(y)*invAngleMultiplier);
	tb->setAngularVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetAngularVelocityZ(GM_Real ID,GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setZ(btScalar(z)*invAngleMultiplier);
	tb->setAngularVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetAngularVelocity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setAngularVelocity(btVector3(btScalar(x),btScalar(y),btScalar(z))*invAngleMultiplier);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_AddAngularVelocity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setAngularVelocity(tb->getAngularVelocity()+btVector3(btScalar(x),btScalar(y),btScalar(z))*invAngleMultiplier);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_ApplyTorqueImpulse(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->applyTorqueImpulse(btVector3(btScalar(x),btScalar(y),btScalar(z))*invAngleMultiplier*invDistanceScale*invDistanceScale);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetLinearVelocityX(GM_Real ID,GM_Real x)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setX(btScalar(x)*invDistanceScale);
	tb->setLinearVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetLinearVelocityY(GM_Real ID,GM_Real y)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setY(btScalar(y)*invDistanceScale);
	tb->setLinearVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetLinearVelocityZ(GM_Real ID,GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getLinearVelocity();
	tv.setZ(btScalar(z)*invDistanceScale);
	tb->setLinearVelocity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetLinearVelocity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{

	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setLinearVelocity(btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_AddLinearVelocity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_ApplyCentralImpulse(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->applyCentralImpulse(btVector3(btScalar(x),btScalar(y),btScalar(z))*invDistanceScale);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_ApplyImpulse(GM_Real ID,GM_Real impulse_x,GM_Real impulse_y,GM_Real impulse_z,GM_Real posx,GM_Real posy,GM_Real posz,GM_Real relative)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=btVector3(btScalar(posx),btScalar(posy),btScalar(posz)*invDistanceScale);
	if(!relative)
		tv-=tb->getCenterOfMassPosition();
	tb->applyImpulse(btVector3(btScalar(impulse_x),btScalar(impulse_y),btScalar(impulse_z))*invDistanceScale,tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_ApplyImpulseFromSpeedAndMass(GM_Real ID,GM_Real velocity_x,GM_Real velocity_y,GM_Real velocity_z,GM_Real mass,GM_Real posx,GM_Real posy,GM_Real posz,GM_Real relative)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=btVector3(btScalar(posx),btScalar(posy),btScalar(posz)*invDistanceScale);
	if(!relative)
		tv-=tb->getCenterOfMassPosition();
	tb->applyImpulse(btVector3(btScalar(velocity_x),btScalar(velocity_y),btScalar(velocity_z))*btScalar(mass),tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_AddVelocityAtPoint(GM_Real ID,GM_Real velocity_x,GM_Real velocity_y,GM_Real velocity_z,GM_Real posx,GM_Real posy,GM_Real posz,GM_Real relative)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=btVector3(btScalar(posx),btScalar(posy),btScalar(posz)*invDistanceScale);
	if(!relative)
		tv-=tb->getCenterOfMassPosition();
	tb->setAngularVelocity(tb->getAngularVelocity()+tv.cross(btVector3(btScalar(velocity_x),btScalar(velocity_y),btScalar(velocity_z))*invDistanceScale));
	tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(btScalar(velocity_x),btScalar(velocity_y),btScalar(velocity_z))*invDistanceScale);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetGravityX(GM_Real ID,GM_Real x)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getGravity();
	tv.setX(btScalar(x));
	tb->setGravity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetGravityY(GM_Real ID,GM_Real y)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getGravity();
	tv.setY(btScalar(y));
	tb->setGravity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetGravityZ(GM_Real ID,GM_Real z)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btVector3 tv=tb->getGravity();
	tv.setZ(btScalar(z));
	tb->setGravity(tv);
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_SetGravity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setGravity(btVector3(btScalar(x),btScalar(y),btScalar(z)));
	tb->activate(1);
	return 0;
}
GM_export GM_Real GMBULLET_AddGravity(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->setGravity(tb->getGravity()+btVector3(btScalar(x),btScalar(y),btScalar(z)));
	tb->activate(1);
	return 0;
}

GM_export GM_Real GMBULLET_SetBodyLinVelFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(tx,ty,tz)*invDistanceScale);
				tb->activate(1);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(btVector3(tx,ty,tz)*invDistanceScale);
				tb->activate(1);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyAngVelFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setAngularVelocity(tb->getAngularVelocity()+btVector3(tx,ty,tz)*invAngleMultiplier);
				tb->activate(1);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setAngularVelocity(btVector3(tx,ty,tz)*invAngleMultiplier);
				tb->activate(1);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyLinAngVelFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=7)
		{
			btScalar trotz=float(ds_stack_pop(stackID));
			btScalar troty=float(ds_stack_pop(stackID));
			btScalar trotx=float(ds_stack_pop(stackID));
			btScalar tpz=float(ds_stack_pop(stackID));
			btScalar tpy=float(ds_stack_pop(stackID));
			btScalar tpx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(tpx,tpy,tpz)*invDistanceScale);
				tb->setAngularVelocity(tb->getAngularVelocity()+btVector3(trotx,troty,trotz)*invAngleMultiplier);
				tb->activate(1);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=7)
		{
			btScalar trotz=float(ds_stack_pop(stackID));
			btScalar troty=float(ds_stack_pop(stackID));
			btScalar trotx=float(ds_stack_pop(stackID));
			btScalar tpz=float(ds_stack_pop(stackID));
			btScalar tpy=float(ds_stack_pop(stackID));
			btScalar tpx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(btVector3(tpx,tpy,tpz)*invDistanceScale);
				tb->setAngularVelocity(btVector3(trotx,troty,trotz)*invAngleMultiplier);
				tb->activate(1);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyGravityFromStack(GM_Real stackID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setGravity(tb->getGravity()+btVector3(tx,ty,tz));
				tb->activate(1);
			}
		}
	}
	else
	{
		while(ds_stack_size(stackID)>=4)
		{
			btScalar tz=float(ds_stack_pop(stackID));
			btScalar ty=float(ds_stack_pop(stackID));
			btScalar tx=float(ds_stack_pop(stackID));
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackID)));
			if(tb)
			{
				retval++;
				tb->setGravity(btVector3(tx,ty,tz));
				tb->activate(1);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyLinVelFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyAngVelFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setAngularVelocity(tb->getAngularVelocity()+btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invAngleMultiplier);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setAngularVelocity(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyLinAngVelFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-7))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(tb->getLinearVelocity()+btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->setAngularVelocity(tb->getAngularVelocity()+btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invAngleMultiplier);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-7))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setLinearVelocity(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->setAngularVelocity(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_SetBodyGravityFromList(GM_Real listID,GM_Real relative)
{
	int retval=0;
	if(relative)
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setGravity(tb->getGravity()+btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	else
	{
		int i=0;
		while(i<=(ds_list_size(listID)-4))
		{
			btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
			if(tb)
			{
				retval++;
				tb->setGravity(btVector3(btScalar(ds_list_find_value(listID,i++)),btScalar(ds_list_find_value(listID,i++)),
						btScalar(ds_list_find_value(listID,i++)))*invDistanceScale);
				tb->activate(1);
			}
			else
			{
				i+=3;
			}
		}
	}
	return retval;
}
// 'Get'Body property Functions
GM_export GM_Real GMBULLET_GetFriction(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getFriction();
}
GM_export GM_Real GMBULLET_IsStatic(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->isStaticObject();
}
GM_export GM_Real GMBULLET_IsKinematic(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->isKinematicObject();
}
GM_export GM_Real GMBULLET_IsStaticOrKinematic(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->isStaticOrKinematicObject();
}
GM_export GM_Real GMBULLET_IsActive(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->isActive();
}
GM_export GM_Real GMBULLET_HasCollided(GM_Real ID)
{
	
		
	gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->m_collided;
}
GM_export GM_Real GMBULLET_GetDeactivationTime(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->getDeactivationTime();
}
GM_export GM_Real GMBULLET_GetAssociatedInstance(GM_Real ID)
{
	
		
	gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->m_GMLIndex;
}
GM_export GM_Real GMBULLET_GetAngularFactor(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getAngularFactor();
}

GM_export GM_Real GMBULLET_GetGravityX(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getGravity().getX()*tb->getInvMass();
}
GM_export GM_Real GMBULLET_GetGravityY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getGravity().getY()*tb->getInvMass();
}
GM_export GM_Real GMBULLET_GetGravityZ(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getGravity().getZ()*tb->getInvMass();
}
GM_export GM_Real GMBULLET_GetPositionX(GM_Real ID)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return (GM_Real)tt.getOrigin().getX()*DistanceScale;
}
GM_export GM_Real GMBULLET_GetPositionY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return (GM_Real)tt.getOrigin().getY()*DistanceScale;
}
GM_export GM_Real GMBULLET_GetPositionZ(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return (GM_Real)tt.getOrigin().getZ()*DistanceScale;
}

GM_export GM_Real GMBULLET_GetRotationAngleAxisX(GM_Real ID)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btQuaternion tq=tt.getRotation();
	if(tq.getW()==1)
		return 1;
	else
		return -tq.getX()/sqrt(1-tq.getW()*tq.getW());
}
GM_export GM_Real GMBULLET_GetRotationAngleAxisY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btQuaternion tq=tt.getRotation();
	if(tq.getW()==1)
		return 0;
	else
		return -tq.getY()/sqrt(1-tq.getW()*tq.getW());
}
GM_export GM_Real GMBULLET_GetRotationAngleAxisZ(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btQuaternion tq=tt.getRotation();
	if(tq.getW()==1)
		return 0;
	else
		return -tq.getZ()/sqrt(1-tq.getW()*tq.getW());
}
GM_export GM_Real GMBULLET_GetRotationAngleAxisTheta(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btQuaternion tq=tt.getRotation();
	return 2*acos(tq.getW())*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetRotationEulerX(GM_Real ID)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btMatrix3x3 tm=tt.getBasis();
	if (tm.getRow(2).x()< +0.999)
	{
		if (tm.getRow(2).x() > -0.999)
			return atan2(-tm.getRow(2).y(),tm.getRow(2).z())*AngleMultiplier;
		else
			return -atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier;
	}
	else
		return atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier;
	//btQuaternion tq=tt.getRotation();
	//btScalar tval=tq.x()*tq.z()+tq.y()*tq.w();
	//if(tval>=0.5)
	//	return 2*atan2(tq.x(),tq.w())*AngleMultiplier;
	//else if(tval<=-0.5)
	//	return -2*atan2(tq.x(),tq.w())*AngleMultiplier;
	//else
	//	return atan2(2*tq.x()*tq.w(),1-2*tq.x()*tq.x()-2*tq.y()*tq.y())*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetRotationEulerY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btMatrix3x3 tm=tt.getBasis();
	if (tm.getRow(2).x()< +0.999)
	{
		if (tm.getRow(2).x() > -0.999)
			return asin(tm.getRow(2).x())*AngleMultiplier;
		else
			return -SIMD_HALF_PI*AngleMultiplier;
	}
	else
		return +SIMD_HALF_PI*AngleMultiplier;
	//btQuaternion tq=tt.getRotation();
	//btScalar tval=tq.x()*tq.z()+tq.y()*tq.w();
	//if(tval>=0.5)
	//	return SIMD_HALF_PI*AngleMultiplier;
	//else if(tval<=-0.5)
	//	return -SIMD_HALF_PI*AngleMultiplier;
	//else
	//	return asin(2*tq.x()*tq.z()+2*tq.y()*tq.w())*AngleMultiplier;
	
}
GM_export GM_Real GMBULLET_GetRotationEulerZ(GM_Real ID)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btMatrix3x3 tm=tt.getBasis();
	if (tm.getRow(2).x()< +0.999 && tm.getRow(2).x() > -0.999)
			return atan2(-tm.getRow(1).x(),tm.getRow(0).x())*AngleMultiplier;
	return 0;
	//btQuaternion tq=tt.getRotation();
	//btScalar tval=tq.x()*tq.z()+tq.y()*tq.w();
	//if(abs(tval)>=0.5)
	//	return 0;
	//else
	//	return atan2(2*tq.z()*tq.w()-2*tq.x()*tq.y(),1-2*tq.y()*tq.y()-2*tq.z()*tq.z())*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetRotationQuaternionX(GM_Real ID)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return tt.getRotation().getX();
}
GM_export GM_Real GMBULLET_GetRotationQuaternionY(GM_Real ID)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return tt.getRotation().getY();
}
GM_export GM_Real GMBULLET_GetRotationQuaternionZ(GM_Real ID)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return tt.getRotation().getZ();
}
GM_export GM_Real GMBULLET_GetRotationQuaternionW(GM_Real ID)
{
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	return tt.getRotation().getW();
}
GM_export GM_Real GMBULLET_GetAngularVelocityX(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getAngularVelocity().getX()*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetAngularVelocityY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getAngularVelocity().getY()*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetAngularVelocityZ(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getAngularVelocity().getZ()*AngleMultiplier;
}
GM_export GM_Real GMBULLET_GetLinearVelocityX(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getLinearVelocity().getX()*DistanceScale;
}
GM_export GM_Real GMBULLET_GetLinearVelocityY(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getLinearVelocity().getY()*DistanceScale;
}
GM_export GM_Real GMBULLET_GetLinearVelocityZ(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getLinearVelocity().getZ()*DistanceScale;
}
GM_export GM_Real GMBULLET_GetCollisionMask(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->getBroadphaseProxy()->m_collisionFilterMask;
}
GM_export GM_Real GMBULLET_GetCollisionGroup(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return (GM_Real)tb->getBroadphaseProxy()->m_collisionFilterGroup;
}

GM_export GM_Real GMBULLET_GetMass(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return GM_Real(1/tb->getInvMass());
}
GM_export GM_Real GMBULLET_GetHitFraction(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return GM_Real(tb->getHitFraction());
}

GM_export GM_Real GMBULLET_GetRestitution(GM_Real ID)
{
	
		
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	return tb->getRestitution();
}
GM_export GM_Real GMBULLET_DeactivationTime(GM_Real ID)
{	
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	tb->getDeactivationTime();
	return 0;
}
GM_export GM_Real GMBULLET_GetDistanceToPoint(GM_Real ID,GM_Real x,GM_Real y,GM_Real z)
{
	gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;
	btTransform tt;
	tb->getMotionState()->getWorldTransform(tt);
	btVector3 tv=btVector3(float(x),float(y),float(z));
	return tv.distance(tt.getOrigin())*DistanceScale;
}
GM_export GM_Real GMBULLET_GetDistanceToBody(GM_Real ID1,GM_Real ID2)
{
	gmbRigidBody* tb1=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID1));;
	gmbRigidBody* tb2=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID2));
	if(!tb1||!tb2)
		return -2;
	btTransform tt;
	tb1->getMotionState()->getWorldTransform(tt);
	btVector3 tv=tt.getOrigin();
	tb2->getMotionState()->getWorldTransform(tt);
	return tv.distance(tt.getOrigin())*DistanceScale;
}


GM_export GM_Real GMBULLET_GetBodyPositionsToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			
			ds_stack_push(stackReturnVals,tt.getOrigin().getZ()*DistanceScale);
			ds_stack_push(stackReturnVals,tt.getOrigin().getY()*DistanceScale);
			ds_stack_push(stackReturnVals,tt.getOrigin().getX()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyRotationsToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			btMatrix3x3 tm=tt.getBasis();

			if (tm.getRow(2).x()< +0.999)
			{
				if (tm.getRow(2).x() > -0.999)
				{
					ds_stack_push(stackReturnVals,atan2(-tm.getRow(1).x(),tm.getRow(0).x())*AngleMultiplier);
					ds_stack_push(stackReturnVals,asin(tm.getRow(2).x())*AngleMultiplier);
					ds_stack_push(stackReturnVals,atan2(-tm.getRow(2).y(),tm.getRow(2).z())*AngleMultiplier);
				}
				else
				{
					ds_stack_push(stackReturnVals,0);
					ds_stack_push(stackReturnVals,-SIMD_HALF_PI*AngleMultiplier);
					ds_stack_push(stackReturnVals,-atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
				}
			}
			else
			{
				ds_stack_push(stackReturnVals,0);
				ds_stack_push(stackReturnVals,+SIMD_HALF_PI*AngleMultiplier);
				ds_stack_push(stackReturnVals,atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
			}
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyTransformsToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			btMatrix3x3 tm=tt.getBasis();

			if (tm.getRow(2).x()< +0.999)
			{
				if (tm.getRow(2).x() > -0.999)
				{
					ds_stack_push(stackReturnVals,atan2(-tm.getRow(1).x(),tm.getRow(0).x())*AngleMultiplier);
					ds_stack_push(stackReturnVals,asin(tm.getRow(2).x())*AngleMultiplier);
					ds_stack_push(stackReturnVals,atan2(-tm.getRow(2).y(),tm.getRow(2).z())*AngleMultiplier);
				}
				else
				{
					ds_stack_push(stackReturnVals,0);
					ds_stack_push(stackReturnVals,-SIMD_HALF_PI*AngleMultiplier);
					ds_stack_push(stackReturnVals,-atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
				}
			}
			else
			{
				ds_stack_push(stackReturnVals,0);
				ds_stack_push(stackReturnVals,+SIMD_HALF_PI*AngleMultiplier);
				ds_stack_push(stackReturnVals,atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
			}

			ds_stack_push(stackReturnVals,tt.getOrigin().getZ()*DistanceScale);
			ds_stack_push(stackReturnVals,tt.getOrigin().getY()*DistanceScale);
			ds_stack_push(stackReturnVals,tt.getOrigin().getX()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyLinVelocitiesToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getZ()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getY()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getX()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyAngVelocitiesToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getZ()*AngleMultiplier);
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getY()*AngleMultiplier);
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getX()*AngleMultiplier);
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyVelocitiesToStack(GM_Real stackBodyIDs,GM_Real stackReturnVals)
{
	int retval=0;
	while(!ds_stack_empty(stackBodyIDs))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_stack_pop(stackBodyIDs)));
		if(tb)
		{
			retval++;
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getZ()*AngleMultiplier);
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getY()*AngleMultiplier);
			ds_stack_push(stackReturnVals,tb->getAngularVelocity().getX()*AngleMultiplier);

			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getZ()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getY()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->getLinearVelocity().getX()*DistanceScale);
			ds_stack_push(stackReturnVals,tb->m_GMLIndex);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyPositionsToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;
			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			
			ds_list_add(listID,tb->m_GMLIndex);
			ds_list_add(listID,tt.getOrigin().getX()*DistanceScale);
			ds_list_add(listID,tt.getOrigin().getY()*DistanceScale);
			ds_list_add(listID,tt.getOrigin().getZ()*DistanceScale);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyRotationsToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;
		
			ds_list_add(listID,tb->m_GMLIndex);

			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			btMatrix3x3 tm=tt.getBasis();

			if (tm.getRow(2).x()< +0.999)
			{
				if (tm.getRow(2).x() > -0.999)
				{
					ds_list_add(listID,atan2(-tm.getRow(2).y(),tm.getRow(2).z())*AngleMultiplier);
					ds_list_add(listID,asin(tm.getRow(2).x())*AngleMultiplier);
					ds_list_add(listID,atan2(-tm.getRow(1).x(),tm.getRow(0).x())*AngleMultiplier);
				}
				else
				{
					ds_list_add(listID,-atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
					ds_list_add(listID,-SIMD_HALF_PI*AngleMultiplier);
					ds_list_add(listID,0);
				}
			}
			else
			{
				ds_list_add(listID,atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
				ds_list_add(listID,+SIMD_HALF_PI*AngleMultiplier);
				ds_list_add(listID,0);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyTransformsToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;
			btTransform tt;
			tb->getMotionState()->getWorldTransform(tt);
			btMatrix3x3 tm=tt.getBasis();

			ds_list_add(listID,tb->m_GMLIndex);
			ds_list_add(listID,tt.getOrigin().getX()*DistanceScale);
			ds_list_add(listID,tt.getOrigin().getY()*DistanceScale);
			ds_list_add(listID,tt.getOrigin().getZ()*DistanceScale);

			if (tm.getRow(2).x()< +0.999)
			{
				if (tm.getRow(2).x() > -0.999)
				{
					ds_list_add(listID,atan2(-tm.getRow(2).y(),tm.getRow(2).z())*AngleMultiplier);
					ds_list_add(listID,asin(tm.getRow(2).x())*AngleMultiplier);
					ds_list_add(listID,atan2(-tm.getRow(1).x(),tm.getRow(0).x())*AngleMultiplier);
				}
				else
				{
					ds_list_add(listID,-atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
					ds_list_add(listID,-SIMD_HALF_PI*AngleMultiplier);
					ds_list_add(listID,0);
				}
			}
			else
			{
				ds_list_add(listID,atan2(tm.getRow(0).y(),tm.getRow(1).y())*AngleMultiplier);
				ds_list_add(listID,+SIMD_HALF_PI*AngleMultiplier);
				ds_list_add(listID,0);
			}
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyLinVelocitiesToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;

			ds_list_add(listID,tb->m_GMLIndex);
			ds_list_add(listID,tb->getLinearVelocity().getX()*DistanceScale);
			ds_list_add(listID,tb->getLinearVelocity().getY()*DistanceScale);
			ds_list_add(listID,tb->getLinearVelocity().getZ()*DistanceScale);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyAngVelocitiesToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;

			ds_list_add(listID,tb->m_GMLIndex);
			ds_list_add(listID,tb->getAngularVelocity().getX()*AngleMultiplier);
			ds_list_add(listID,tb->getAngularVelocity().getY()*AngleMultiplier);
			ds_list_add(listID,tb->getAngularVelocity().getZ()*AngleMultiplier);
		}
	}
	return retval;
}
GM_export GM_Real GMBULLET_GetBodyVelocitiesToList(GM_Real listID)
{
	int retval=0;
	int i=0;
	while(i<ds_list_size(listID))
	{
		gmbRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ds_list_find_value(listID,i++)));
		if(tb)
		{
			retval++;

			ds_list_add(listID,tb->m_GMLIndex);

			ds_list_add(listID,tb->getLinearVelocity().getX()*DistanceScale);
			ds_list_add(listID,tb->getLinearVelocity().getY()*DistanceScale);
			ds_list_add(listID,tb->getLinearVelocity().getZ()*DistanceScale);

			ds_list_add(listID,tb->getAngularVelocity().getX()*AngleMultiplier);
			ds_list_add(listID,tb->getAngularVelocity().getY()*AngleMultiplier);
			ds_list_add(listID,tb->getAngularVelocity().getZ()*AngleMultiplier);
		}
	}
	return retval;
}