#include "gmbMiscExports.h"
#define _USE_MATH_DEFINES 1
#include <math.h>
#include "GMFunctions.h"

//Bullet class variables
btDiscreteDynamicsWorld* gmbGlobal_World=NULL;
btCollisionDispatcher* gmbGlobal_Dispatcher=NULL;
btBroadphaseInterface* gmbGlobal_Broadphase=NULL;
btSequentialImpulseConstraintSolver* gmbGlobal_ConstraintSolver=NULL;

gmbMyOverlappingPairCallback* gmbGlobal_myOverPairCall=NULL;

btScalar DistanceScale=1; //Scaling
btScalar invDistanceScale=1;
btScalar AngleMultiplier=1; //Convert to left/right coordinate system and deg/rad
btScalar invAngleMultiplier=1;

bool gmbGlobal_IsInitialized=false;

btCollisionConfiguration* gmbLocalGlobal_CollisionInfo=NULL;
btAlignedObjectArray<gmbRigidBody*> gmbLocalGlobal_List;

//Rotation With Flags Functions
//These 6 functions are each for a rotation about 1 of the 3 axis,
//and whether or not to apply it to the model's axis or the World axis
//Capital axis letter indicates the function is applied to the model's current axis, wheras lower
//case denotes that it is applied relative to the World axis.
//These are then pointed to by the variables RotationWithFlags1func, 2 and 3, which allows rotations
//of any order to be specified. RotationWithFlags is the main function which calls the other 3.
void RotationWithFlagsX(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(tm->getColumn(0),rot));
}
void RotationWithFlagsx(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(btVector3(1,0,0),rot));
}
void RotationWithFlagsY(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(tm->getColumn(1),rot));
}
void RotationWithFlagsy(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(btVector3(0,1,0),rot));
}
void RotationWithFlagsZ(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(tm->getColumn(2),rot));
}
void RotationWithFlagsz(btScalar rot,btMatrix3x3 *tm)
{
	(*tm)*=btMatrix3x3(btQuaternion(btVector3(0,0,1),rot));
}
void (*RotationWithFlags1func)(float, btMatrix3x3*)=RotationWithFlagsx;
void (*RotationWithFlags2func)(float, btMatrix3x3*)=RotationWithFlagsy;
void (*RotationWithFlags3func)(float, btMatrix3x3*)=RotationWithFlagsz;

btMatrix3x3 RotationWithFlags(btScalar r1,btScalar r2,btScalar r3,btMatrix3x3 &tm)
{
	RotationWithFlags1func(r1*invAngleMultiplier,&tm);
	RotationWithFlags2func(r2*invAngleMultiplier,&tm);
	RotationWithFlags3func(r3*invAngleMultiplier,&tm);
	return tm;
}
//-----------------------------

void MyNearCallback(btBroadphasePair& collisionPair,btCollisionDispatcher& dispatcher,btDispatcherInfo &dispatchInfo)
{
	gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)collisionPair.m_pProxy0->m_clientObject);
	if(tb)
		tb->m_collided=true;
	tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)collisionPair.m_pProxy1->m_clientObject);
	if(tb)
		tb->m_collided=true;
	dispatcher.defaultNearCallback(collisionPair, dispatcher, dispatchInfo);
}
//----------------------------
//World Functions
GM_export GM_Real GMBULLET_CreateBulletWorld(GM_Real minx,GM_Real miny,GM_Real minz,GM_Real maxx,GM_Real maxy,GM_Real maxz,GM_Real broadphaseType)
{
	if(gmbGlobal_IsInitialized)
		return -1;

	gmbLocalGlobal_CollisionInfo=new btDefaultCollisionConfiguration();
	gmbGlobal_Dispatcher=new btCollisionDispatcher(gmbLocalGlobal_CollisionInfo);
	btGImpactCollisionAlgorithm::registerAlgorithm(gmbGlobal_Dispatcher);
	gmbGlobal_Dispatcher->setNearCallback(btNearCallback(MyNearCallback));


	gmbGlobal_myOverPairCall=new gmbMyOverlappingPairCallback();

	//Broadphase
	btVector3 tWorldAabbMin;
	tWorldAabbMin.setValue(btScalar(minx),btScalar(miny),btScalar(minz));
	btVector3 tWorldAabbMax;
	tWorldAabbMax.setValue(btScalar(maxx),btScalar(maxy),btScalar(maxz));

	if(broadphaseType==0)
	{
		btAxisSweep3* tc=new btAxisSweep3(tWorldAabbMin,tWorldAabbMax);
		tc->setOverlappingPairUserCallback(gmbGlobal_myOverPairCall);
		gmbGlobal_Broadphase=tc;
	}
	else //if(broadphaseType==1)
	{
		bt32BitAxisSweep3* tc=new bt32BitAxisSweep3(tWorldAabbMin,tWorldAabbMax);
		tc->setOverlappingPairUserCallback(gmbGlobal_myOverPairCall);
		gmbGlobal_Broadphase=tc;
	}
	/*else if(broadphaseType==2)
	{
		bt32BitAxisSweep3* tc=new bt32BitAxisSweep3(tWorldAabbMin,tWorldAabbMax);
		tc->setOverlappingPairUserCallback(tMyOverPairCall);
		gmbGlobal_Broadphase=tc;
	}*/

	gmbGlobal_ConstraintSolver = new btSequentialImpulseConstraintSolver();  //Constraint Solver
	

	gmbGlobal_World=new btDiscreteDynamicsWorld(gmbGlobal_Dispatcher,gmbGlobal_Broadphase,gmbGlobal_ConstraintSolver,gmbLocalGlobal_CollisionInfo);

	gmbGlobal_IsInitialized=true;
	return 0;
}
GM_export GM_Real GMBULLET_DestroyBulletWorld()
{
	if(!gmbGlobal_IsInitialized)
		return -1;

	while(gmbGlobal_Constraints.m_indexArray.size())
	{
		GMBULLET_DeleteConstraint(gmbGlobal_Constraints.m_indexArray.size()-1);
	}
	while(gmbGlobal_RigidBodies.m_indexArray.size())
	{
		GMBULLET_DeleteRigidBody(gmbGlobal_RigidBodies.m_indexArray.size()-1);
	}

	gmbGlobal_IsInitialized=false;

	if(gmbGlobal_World)
		delete gmbGlobal_World;
	gmbGlobal_World=NULL;
	if(gmbGlobal_ConstraintSolver)
		delete gmbGlobal_ConstraintSolver;
	gmbGlobal_ConstraintSolver=NULL;
	if(gmbGlobal_Broadphase)
		delete gmbGlobal_Broadphase;
	gmbGlobal_Broadphase=NULL;
	if(gmbGlobal_myOverPairCall)
		delete gmbGlobal_myOverPairCall;
	gmbGlobal_myOverPairCall=NULL;
	if(gmbGlobal_Dispatcher)
		delete gmbGlobal_Dispatcher;
	gmbGlobal_Dispatcher=NULL;
	if(gmbLocalGlobal_CollisionInfo)
		delete gmbLocalGlobal_CollisionInfo;
	gmbLocalGlobal_CollisionInfo=NULL;
	if(gmbGlobal_myOverPairCall)
		delete gmbGlobal_myOverPairCall;
	gmbGlobal_myOverPairCall=NULL;

	return 0;
}
GM_export GM_Real GMBULLET_StepSimulationTime(GM_Real timestep,GM_Real maxsteps,GM_Real fixedTimestep)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	int t=gmbGlobal_World->getNumCollisionObjects();
	for(int i=0;i<t;i++)
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>(gmbGlobal_World->getCollisionObjectArray()[i]);
		if(!tb)
			continue;
		tb->m_collided=false;
	}
	gmbGlobal_myOverPairCall->clear();
	gmbGlobal_World->stepSimulation(float(timestep),int(maxsteps),float(fixedTimestep));
	return 0;
}
GM_export GM_Real GMBULLET_SetWorldGravity(GM_Real x,GM_Real y,GM_Real z)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbGlobal_World->setGravity(btVector3(float(x),float(y),float(z)));
	return 0;
}

GM_export GM_Real GMBULLET_SetWorldScale(GM_Real scale)
{
	DistanceScale=float(scale);
	if(!DistanceScale)
		return -1;
	invDistanceScale=1/DistanceScale;
	return 0;
}
GM_export GM_Real GMBULLET_SetAngleProperties(GM_Real isDegrees,GM_Real isLeftHanded)
{
	if(isLeftHanded)
		AngleMultiplier=1;
	else
		AngleMultiplier=-1;
	if(isDegrees)
		AngleMultiplier*=float((M_1_PI*180));
	if(!AngleMultiplier)
		return -1;
	invAngleMultiplier=1/AngleMultiplier;
	return 0;
}
GM_export GM_Real GMBULLET_SetRotationFlags(GM_Real axis1,GM_Real axis2,GM_Real axis3)
{
	switch((int)axis1)
	{
	case 0:
		RotationWithFlags1func=RotationWithFlagsx;
		break;
	case 1:
		RotationWithFlags1func=RotationWithFlagsy;
		break;
	case 2:
		RotationWithFlags1func=RotationWithFlagsz;
		break;
	case 3:
		RotationWithFlags1func=RotationWithFlagsX;
		break;
	case 4:
		RotationWithFlags1func=RotationWithFlagsY;
		break;
	case 5:
		RotationWithFlags1func=RotationWithFlagsZ;
		break;
	default:
		RotationWithFlags1func=RotationWithFlagsx;
	}
	switch((int)axis2)
	{
	case 0:
		RotationWithFlags2func=RotationWithFlagsx;
		break;
	case 1:
		RotationWithFlags2func=RotationWithFlagsy;
		break;
	case 2:
		RotationWithFlags2func=RotationWithFlagsz;
		break;
	case 3:
		RotationWithFlags2func=RotationWithFlagsX;
		break;
	case 4:
		RotationWithFlags2func=RotationWithFlagsY;
		break;
	case 5:
		RotationWithFlags2func=RotationWithFlagsZ;
		break;
	default:
		RotationWithFlags2func=RotationWithFlagsy;
	}
	switch((int)axis3)
	{
	case 0:
		RotationWithFlags3func=RotationWithFlagsx;
		break;
	case 1:
		RotationWithFlags3func=RotationWithFlagsy;
		break;
	case 2:
		RotationWithFlags3func=RotationWithFlagsz;
		break;
	case 3:
		RotationWithFlags3func=RotationWithFlagsX;
		break;
	case 4:
		RotationWithFlags3func=RotationWithFlagsY;
		break;
	case 5:
		RotationWithFlags3func=RotationWithFlagsZ;
		break;
	default:
		RotationWithFlags3func=RotationWithFlagsy;
	}
	return 0;
}
//-----------------------

//-----------------------
//Collision Tests
struct Rayresult
{
	float hitpointX,hitpointY,hitpointZ,
		hitnormalX,hitnormalY,hitnormalZ,
		hitfraction;
	int objectID;
	unsigned int GMID;
};
struct gmbClosestRayResultCallback : btCollisionWorld::ClosestRayResultCallback
{
	gmbClosestRayResultCallback(const btVector3&	rayFromWorld,const btVector3&	rayToWorld,btCollisionObject* excludeObj):ClosestRayResultCallback(rayFromWorld,rayToWorld),ignoreObj(excludeObj)
	{
	}

	btCollisionObject* ignoreObj;

	virtual	btScalar addSingleResult(btCollisionWorld::LocalRayResult& rayResult,bool normalInWorldSpace)
	{
		if(rayResult.m_collisionObject==ignoreObj)
			return rayResult.m_hitFraction;
		else
			return ClosestRayResultCallback::addSingleResult(rayResult,normalInWorldSpace);
	}
};
struct AllRayresultCallback : public btCollisionWorld::RayResultCallback
{
	AllRayresultCallback(const btVector3&	rayFromWorld,const btVector3&	rayToWorld,btCollisionObject* excludeObj):m_rayFromWorld(rayFromWorld),m_rayToWorld(rayToWorld)
	{
		m_closestHitFraction=2;
		ignoreObj=excludeObj;
	}

	btVector3	m_rayFromWorld;
	btVector3	m_rayToWorld;

	btVector3	m_hitNormalWorld;
	btVector3	m_hitPointWorld;

	btAlignedObjectArray<Rayresult> Rayresults;
	btCollisionObject* ignoreObj;

	virtual	btScalar	addSingleResult(btCollisionWorld::LocalRayResult& rayResult,bool normalInWorldSpace)
	{
		Rayresult tr;

		tr.hitfraction=rayResult.m_hitFraction;
		if(ignoreObj==rayResult.m_collisionObject)
			return rayResult.m_hitFraction;

		m_collisionObject = rayResult.m_collisionObject;
		if (normalInWorldSpace)
		{
			m_hitNormalWorld = rayResult.m_hitNormalLocal;
		} else
		{
			///need to transform normal into worldspace
			m_hitNormalWorld = m_collisionObject->getWorldTransform().getBasis()*rayResult.m_hitNormalLocal;
		}
		m_hitPointWorld.setInterpolate3(m_rayFromWorld,m_rayToWorld,rayResult.m_hitFraction);

		tr.hitnormalX=m_hitNormalWorld.getX();
		tr.hitnormalY=m_hitNormalWorld.getY();
		tr.hitnormalZ=m_hitNormalWorld.getZ();

		tr.hitpointX=m_hitPointWorld.getX();
		tr.hitpointY=m_hitPointWorld.getY();
		tr.hitpointZ=m_hitPointWorld.getZ();

		tr.objectID=reinterpret_cast<gmbRigidBody*>(m_collisionObject)->m_GMBIndex;
		tr.GMID=reinterpret_cast<gmbRigidBody*>(m_collisionObject)->m_GMLIndex;

		Rayresults.push_back(tr);

		return rayResult.m_hitFraction;
	}
};
struct gmbClosestConvexResultCallback : public btCollisionWorld::ClosestConvexResultCallback
{
	gmbClosestConvexResultCallback(const btVector3&	convexFromWorld,const btVector3&	convexToWorld,btCollisionObject* excludeObj)
	:ClosestConvexResultCallback(convexFromWorld,convexToWorld),ignoreObj(excludeObj)
	{
	}

	btCollisionObject* ignoreObj;

	virtual	btScalar addSingleResult(btCollisionWorld::LocalConvexResult& convexResult,bool normalInWorldSpace)
	{
		if(ignoreObj==convexResult.m_hitCollisionObject)
			return convexResult.m_hitFraction;
		else
			return ClosestConvexResultCallback::addSingleResult(convexResult,normalInWorldSpace);
	}
};
struct	AllConvexResultCallback : public btCollisionWorld::ConvexResultCallback
{
	AllConvexResultCallback(const btVector3& convexFromWorld,const btVector3& convexToWorld,btCollisionObject* excludeObj):m_convexFromWorld(convexFromWorld),m_convexToWorld(convexToWorld),m_hitCollisionObject(0)
	{
		ignoreObj=excludeObj;
	}

	btVector3	m_convexFromWorld;//used to calculate hitPointWorld from hitFraction
	btVector3	m_convexToWorld;

	btVector3	m_hitNormalWorld;
	btVector3	m_hitPointWorld;
	btCollisionObject*	m_hitCollisionObject,*ignoreObj;
	btAlignedObjectArray<Rayresult> Convexresults;


	virtual	btScalar addSingleResult(btCollisionWorld::LocalConvexResult& convexResult,bool normalInWorldSpace)
	{
		if(convexResult.m_hitCollisionObject==ignoreObj)
			return convexResult.m_hitFraction;

		Rayresult tr;
		tr.hitfraction=convexResult.m_hitFraction;

		m_hitCollisionObject = convexResult.m_hitCollisionObject;
		if (normalInWorldSpace)
		{
			m_hitNormalWorld = convexResult.m_hitNormalLocal;
		} else
		{
			///need to transform normal into worldspace
			m_hitNormalWorld = m_hitCollisionObject->getWorldTransform().getBasis()*convexResult.m_hitNormalLocal;
		}
		m_hitPointWorld = convexResult.m_hitPointLocal;

		tr.hitnormalX=m_hitNormalWorld.getX();
		tr.hitnormalY=m_hitNormalWorld.getY();
		tr.hitnormalZ=m_hitNormalWorld.getZ();

		tr.hitpointX=m_hitPointWorld.getX();
		tr.hitpointY=m_hitPointWorld.getY();
		tr.hitpointZ=m_hitPointWorld.getZ();

		tr.objectID=reinterpret_cast<gmbRigidBody*>(m_hitCollisionObject)->m_GMBIndex;
		tr.GMID=reinterpret_cast<gmbRigidBody*>(m_hitCollisionObject)->m_GMLIndex;

		Convexresults.push_back(tr);

		return convexResult.m_hitFraction;
	}
};
class gmbRayResultSort
{
	public:

		bool operator() ( Rayresult &a, Rayresult &b )
		{
			 return a.hitfraction>b.hitfraction;
		}
};
btAlignedObjectArray<Rayresult> RayConvexresults;
//Ray Tests
GM_export GM_Real GMBULLET_RayTestClosestHit(GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto,GM_Real cMask,GM_Real cGroup,GM_Real retInfo,GM_Real ignore_ID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;

	btVector3 trfrom=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale;
	btVector3 trto=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;

	btCollisionObject* ignoreObj;
	if(ignore_ID<0)
		ignoreObj=NULL;
	else
		ignoreObj=gmbGlobal_RigidBodies.GetInstance(unsigned int(ignore_ID));

	gmbClosestRayResultCallback result(trfrom,trto,ignoreObj);
	result.m_collisionFilterMask=short(int(cMask));
	result.m_collisionFilterGroup=short(int(cGroup));

	gmbGlobal_World->rayTest(trfrom,trto,result);
	RayConvexresults.clear();
	if(!result.hasHit())
	{
		if(retInfo)
			return -2;
		else
			return 0;
	}
	Rayresult tresult;
	tresult.hitfraction=result.m_closestHitFraction;
	tresult.hitnormalX=result.m_hitNormalWorld.getX();
	tresult.hitnormalY=result.m_hitNormalWorld.getY();
	tresult.hitnormalZ=result.m_hitNormalWorld.getZ();
	tresult.hitpointX=result.m_hitPointWorld.getX();
	tresult.hitpointY=result.m_hitPointWorld.getY();
	tresult.hitpointZ=result.m_hitPointWorld.getZ();
	tresult.objectID=reinterpret_cast<gmbRigidBody*>(result.m_collisionObject)->m_GMBIndex;
	tresult.GMID=reinterpret_cast<gmbRigidBody*>(result.m_collisionObject)->m_GMLIndex;

	RayConvexresults.push_back(tresult);
	switch(int(retInfo))
	{
	case 1:
		return (double)tresult.objectID;
		break;
	case 2:
		return (double)tresult.GMID;
		break;
	default:
		return 1;
	}
}
GM_export GM_Real GMBULLET_RayTestAllHit(GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto,GM_Real cMask,GM_Real cGroup,GM_Real ignore_ID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	btVector3 rfrom=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale;
	btVector3 rto=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;
	btCollisionObject* ignoreObj;
	if(ignore_ID<0)
		ignoreObj=NULL;
	else
		ignoreObj=gmbGlobal_RigidBodies.GetInstance(unsigned int(ignore_ID));
	AllRayresultCallback result(rfrom,rto,ignoreObj);
	result.m_collisionFilterMask=short(int(cMask));
	result.m_collisionFilterGroup=short(int(cGroup));

	gmbGlobal_World->rayTest(rfrom,rto,result);
	RayConvexresults.clear();
	RayConvexresults=result.Rayresults;
	RayConvexresults.heapSort(gmbRayResultSort());
	return RayConvexresults.size();
}
GM_export GM_Real GMBULLET_RayTestSingle(GM_Real ID,GM_Real ShapeID,GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	btRigidBody* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tb)
		return -2;

	btVector3 rfrom=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale;
	btVector3 rto=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;
	btCollisionWorld::ClosestRayResultCallback result(rfrom,rto);

	btTransform tt1,tt2;
	tt1.setIdentity();
	tt1.setOrigin(rfrom);
	tt2.setIdentity();
	tt2.setOrigin(rto);

	btCollisionShape* trcs;
	gmbShape* tgmbcs=gmbGlobal_ShapeList.GetInstance(unsigned int(ShapeID));
	if(tgmbcs)
	{
		trcs=tgmbcs->getShape();
	}
	else
	{
		trcs=tb->getCollisionShape();
	}

	gmbGlobal_World->rayTestSingle(tt1,tt2,tb,trcs,tb->getWorldTransform(),result);

	RayConvexresults.clear();
	if(!result.hasHit())
		return 0;
	Rayresult tresult;

	tresult.hitfraction=result.m_closestHitFraction;
	tresult.hitnormalX=result.m_hitNormalWorld.getX();
	tresult.hitnormalY=result.m_hitNormalWorld.getY();
	tresult.hitnormalZ=result.m_hitNormalWorld.getZ();
	tresult.hitpointX=result.m_hitPointWorld.getX();
	tresult.hitpointY=result.m_hitPointWorld.getY();
	tresult.hitpointZ=result.m_hitPointWorld.getZ();
	tresult.objectID=reinterpret_cast<gmbRigidBody*>(result.m_collisionObject)->m_GMBIndex;
	tresult.GMID=reinterpret_cast<gmbRigidBody*>(result.m_collisionObject)->m_GMLIndex;

	RayConvexresults.push_back(tresult);
	return RayConvexresults.size();
}

//Convex Tests
btMatrix3x3 ConvexSweepOrientationStart,ConvexSweepOrientationEnd;
GM_export GM_Real GMBULLET_ConvexSweepOrientation(GM_Real rx1,GM_Real ry1,GM_Real rz1,GM_Real rx2,GM_Real ry2,GM_Real rz2)
{
	ConvexSweepOrientationStart=RotationWithFlags(float(rx1),float(ry1),float(rz1));
	ConvexSweepOrientationEnd=RotationWithFlags(float(rx2),float(ry2),float(rz2));
	return 0;
}
GM_export GM_Real GMBULLET_ConvexSweepClosestHit(GM_Real ShapeID,GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto,GM_Real cMask,GM_Real cGroup,GM_Real retInfo,GM_Real ignore_ID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbShape* tgmbs=gmbGlobal_ShapeList.GetInstance(unsigned int(ShapeID));
	if(!tgmbs)
		return -1;
	btCollisionShape* ts=tgmbs->getShape();
	if(!ts->isConvex())
		return -1;
	btConvexShape* tcs=static_cast<btConvexShape*>(ts);

	btVector3 from=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale,
			  to=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;

	btCollisionObject* ignoreObj;
	if(ignore_ID<0)
		ignoreObj=NULL;
	else
		ignoreObj=gmbGlobal_RigidBodies.GetInstance(unsigned int(ignore_ID));

	gmbClosestConvexResultCallback ResultCallback(from,to,ignoreObj);
	ResultCallback.m_collisionFilterMask=short(int(cMask));
	ResultCallback.m_collisionFilterGroup=short(int(cGroup));

	gmbGlobal_World->convexSweepTest(tcs,btTransform(ConvexSweepOrientationStart,from),btTransform(ConvexSweepOrientationEnd,to),ResultCallback,short(cMask));

	RayConvexresults.clear();
	if(!ResultCallback.hasHit())
	{
		if(retInfo)
			return -2;
		else
			return 0;
	}

	Rayresult tresult;

	tresult.hitfraction=ResultCallback.m_closestHitFraction;
	tresult.hitnormalX=ResultCallback.m_hitNormalWorld.getX();
	tresult.hitnormalY=ResultCallback.m_hitNormalWorld.getY();
	tresult.hitnormalZ=ResultCallback.m_hitNormalWorld.getZ();
	tresult.hitpointX=ResultCallback.m_hitPointWorld.getX();
	tresult.hitpointY=ResultCallback.m_hitPointWorld.getY();
	tresult.hitpointZ=ResultCallback.m_hitPointWorld.getZ();
	tresult.objectID=reinterpret_cast<gmbRigidBody*>(ResultCallback.m_hitCollisionObject)->m_GMBIndex;
	tresult.GMID=reinterpret_cast<gmbRigidBody*>(ResultCallback.m_hitCollisionObject)->m_GMLIndex;

	RayConvexresults.push_back(tresult);
	switch(int(retInfo))
	{
	case 1:
		return (double)tresult.objectID;
		break;
	case 2:
		return (double)tresult.GMID;
		break;
	default:
		return 1;
	}
}
GM_export GM_Real GMBULLET_ConvexSweepAllHit(GM_Real ShapeID,GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto,GM_Real cMask,GM_Real cGroup,GM_Real ignore_ID)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbShape* tgmbs=gmbGlobal_ShapeList.GetInstance(unsigned int(ShapeID));
	if(!tgmbs)
		return -1;
	btCollisionShape* ts=tgmbs->getShape();
	if(!ts->isConvex())
		return -1;
	btConvexShape* tcs=static_cast<btConvexShape*>(ts);

	btVector3 from=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale,
			  to=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;

	btCollisionObject* ignoreObj;
	if(ignore_ID<0)
		ignoreObj=NULL;
	else
		ignoreObj=gmbGlobal_RigidBodies.GetInstance(unsigned int(ignore_ID));

	AllConvexResultCallback ResultCallback(from,to,ignoreObj);
	ResultCallback.m_collisionFilterMask=short(int(cMask));
	ResultCallback.m_collisionFilterGroup=short(int(cGroup));

	gmbGlobal_World->convexSweepTest(tcs,btTransform(ConvexSweepOrientationStart,from),btTransform(ConvexSweepOrientationEnd,to),ResultCallback,(short)cMask);
	RayConvexresults.clear();
	RayConvexresults=ResultCallback.Convexresults;
	RayConvexresults.heapSort(gmbRayResultSort());
	return RayConvexresults.size();
}
GM_export GM_Real GMBULLET_ConvexSweepSingle(GM_Real ShapeID,GM_Real BodyID,GM_Real ShapeID2,GM_Real xfrom,GM_Real yfrom,GM_Real zfrom,GM_Real xto,GM_Real yto,GM_Real zto)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbShape* tgmbs=gmbGlobal_ShapeList.GetInstance(unsigned int(ShapeID));
	if(!tgmbs)
		return -1;
	btCollisionShape* ts=tgmbs->getShape();
	if(!ts->isConvex())
		return -1;
	btCollisionObject* tb=gmbGlobal_RigidBodies.GetInstance(unsigned int(BodyID));
	if(!tb)
		return -1;

	btConvexShape* tcs=static_cast<btConvexShape*>(ts);

	btCollisionShape* trcs;
	gmbShape* tgmbcs=gmbGlobal_ShapeList.GetInstance(unsigned int(ShapeID2));
	if(tgmbcs)
	{
		trcs=tgmbcs->getShape();
	}
	else
	{
		trcs=tb->getCollisionShape();
	}

	btVector3 from=btVector3(float(xfrom),float(yfrom),float(zfrom))*invDistanceScale,
			  to=btVector3(float(xto),float(yto),float(zto))*invDistanceScale;
	btCollisionWorld::ClosestConvexResultCallback ResultCallback(from,to);

	gmbGlobal_World->objectQuerySingle(tcs,btTransform(ConvexSweepOrientationStart,from),btTransform(ConvexSweepOrientationEnd,to),tb,trcs,tb->getWorldTransform(),ResultCallback,10);

	RayConvexresults.clear();
	if(!ResultCallback.hasHit())
		return 0;
	Rayresult tresult;

	tresult.hitfraction=ResultCallback.m_closestHitFraction;
	tresult.hitnormalX=ResultCallback.m_hitNormalWorld.getX();
	tresult.hitnormalY=ResultCallback.m_hitNormalWorld.getY();
	tresult.hitnormalZ=ResultCallback.m_hitNormalWorld.getZ();
	tresult.hitpointX=ResultCallback.m_hitPointWorld.getX();
	tresult.hitpointY=ResultCallback.m_hitPointWorld.getY();
	tresult.hitpointZ=ResultCallback.m_hitPointWorld.getZ();
	tresult.objectID=reinterpret_cast<gmbRigidBody*>(ResultCallback.m_hitCollisionObject)->m_GMBIndex;
	tresult.GMID=reinterpret_cast<gmbRigidBody*>(ResultCallback.m_hitCollisionObject)->m_GMLIndex;

	RayConvexresults.push_back(tresult);
	return RayConvexresults.size();
}

//Collision Test results
GM_export GM_Real GMBULLET_GetRayConvexTestHitPointX(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitpointX;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitPointY(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitpointY;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitPointZ(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitpointZ;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitNormalX(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitnormalX;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitNormalY(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitnormalY;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitNormalZ(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitnormalZ;
}
GM_export GM_Real GMBULLET_GetRayConvexTestHitFraction(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].hitfraction;
}
GM_export GM_Real GMBULLET_GetRayConvexTestObjectID(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].objectID;
}
GM_export GM_Real GMBULLET_GetRayConvexTestGMID(GM_Real num)
{
	unsigned int tnum=(unsigned int)num;
	if(!(num<RayConvexresults.size()))
		return -2;
	return RayConvexresults[tnum].GMID;
}
GM_export GM_Real GMBULLET_ClearRayConvexTestResults()
{
	RayConvexresults.clear();
	return 0;
}
//Test using Data Structures
void PushTestResultsOntoDSStack(double stackID,int numTests)
{
	for(int i=(numTests-1);i>=0;--i)
	{
		ds_stack_push(stackID,RayConvexresults[i].hitfraction);
		ds_stack_push(stackID,RayConvexresults[i].hitnormalZ);
		ds_stack_push(stackID,RayConvexresults[i].hitnormalY);
		ds_stack_push(stackID,RayConvexresults[i].hitnormalX);
		ds_stack_push(stackID,RayConvexresults[i].hitpointZ);
		ds_stack_push(stackID,RayConvexresults[i].hitpointY);
		ds_stack_push(stackID,RayConvexresults[i].hitpointX);
		ds_stack_push(stackID,RayConvexresults[i].objectID);
		ds_stack_push(stackID,RayConvexresults[i].GMID);
	}
	ds_stack_push(stackID,numTests);
}
GM_export GM_Real GMBULLET_PerformTestUsingDSStack(GM_Real inputQueueID,GM_Real resultsStackID)
{
	int retval=0;
	while(!ds_queue_empty(inputQueueID))
	{
		switch(int(ds_queue_dequeue(inputQueueID)))
		{
		case 1: //Ray test closest hit
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestClosestHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),0,
						ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 2: //Ray test all hit
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestAllHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 3: //Ray test single
			{
				if(ds_queue_size(inputQueueID)<8)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestSingle(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 4: //Convex sweep closest hit
			{
				if(ds_queue_size(inputQueueID)<10)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepClosestHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						0,ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 5: //Convex sweep all hit
			{
				if(ds_queue_size(inputQueueID)<10)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepAllHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 6: //Convex sweep single
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepSingle(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSStack(resultsStackID,ret);
			}
		case 7: //Convex sweep orientation
			{
				if(ds_queue_size(inputQueueID)<6)
					return retval;
				retval++;
				GMBULLET_ConvexSweepOrientation(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID));
			}
		}
	}
	return retval;
}
void PushTestResultsOntoDSList(double listID,int numTests)
{
	for(int i=(numTests-1);i>=0;--i)
	{
		ds_list_add(listID,numTests);
		ds_list_add(listID,RayConvexresults[i].GMID);
		ds_list_add(listID,RayConvexresults[i].objectID);
		ds_list_add(listID,RayConvexresults[i].hitpointX);
		ds_list_add(listID,RayConvexresults[i].hitpointY);
		ds_list_add(listID,RayConvexresults[i].hitpointZ);
		ds_list_add(listID,RayConvexresults[i].hitnormalX);
		ds_list_add(listID,RayConvexresults[i].hitnormalY);
		ds_list_add(listID,RayConvexresults[i].hitnormalZ);
		ds_list_add(listID,RayConvexresults[i].hitfraction);
	}
}
GM_export GM_Real GMBULLET_PerformTestUsingDSList(GM_Real inputQueueID,GM_Real resultsListID)
{
	int retval=0;
	while(!ds_queue_empty(inputQueueID))
	{
		switch(int(ds_queue_dequeue(inputQueueID)))
		{
		case 1: //Ray test closest hit
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestClosestHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),0,
						ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 2: //Ray test all hit
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestAllHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 3: //Ray test single
			{
				if(ds_queue_size(inputQueueID)<8)
					return retval;
				retval++;
				int ret=int(GMBULLET_RayTestSingle(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 4: //Convex sweep closest hit
			{
				if(ds_queue_size(inputQueueID)<10)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepClosestHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						0,ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 5: //Convex sweep all hit
			{
				if(ds_queue_size(inputQueueID)<10)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepAllHit(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 6: //Convex sweep single
			{
				if(ds_queue_size(inputQueueID)<9)
					return retval;
				retval++;
				int ret=int(GMBULLET_ConvexSweepSingle(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID)));
				PushTestResultsOntoDSList(resultsListID,ret);
			}
		case 7: //Convex sweep orientation
			{
				if(ds_queue_size(inputQueueID)<6)
					return retval;
				retval++;
				GMBULLET_ConvexSweepOrientation(ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),
						ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID),ds_queue_dequeue(inputQueueID));
			}
		}
	}
	return retval;
}
//-----------------------
//Active list, and Collision Pair list
GM_export GM_Real GMBULLET_GenerateActiveList(GM_Real excludeWantsDeactivation)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbLocalGlobal_List.clear();
	int t=gmbGlobal_World->getNumCollisionObjects();
	if(!excludeWantsDeactivation)
	{
		for(int i=0;i<t;i++)
		{
			gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>(gmbGlobal_World->getCollisionObjectArray()[i]);
			if(tb)
			{
				if(tb->isActive())
					gmbLocalGlobal_List.push_back(tb);
			}
		}
	}
	else
	{
		for(int i=0;i<t;i++)
		{
			gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>(gmbGlobal_World->getCollisionObjectArray()[i]);
			if(tb)
			{
				if(tb->getActivationState()==1||tb->getActivationState()==4)
					gmbLocalGlobal_List.push_back(tb);
			}
		}
	}
	return gmbLocalGlobal_List.size();
}
GM_export GM_Real GMBULLET_GenerateCollidedList()
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	gmbLocalGlobal_List.clear();
	int t=gmbGlobal_World->getNumCollisionObjects();
	for(int i=0;i<t;i++)
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>(gmbGlobal_World->getCollisionObjectArray()[i]);
		if(tb)
		{
			if (tb->m_collided)
				gmbLocalGlobal_List.push_back(tb);
		}
	}
	return gmbLocalGlobal_List.size();
}
GM_export GM_Real GMBULLET_GenerateCollideWithBodyList(GM_Real ID,GM_Real type)
{
	btRigidBody* tgb=gmbGlobal_RigidBodies.GetInstance(unsigned int(ID));
	if(!tgb)
		return -1;
	btBroadphasePairArray* tpc;
	switch(int(type))
	{
	case 0:
		tpc=&gmbGlobal_World->getPairCache()->getOverlappingPairArray();
	case 1:
		tpc=&gmbGlobal_myOverPairCall->newPair;
	case 2:
		tpc=&gmbGlobal_myOverPairCall->oldPair;
	default: return -1;
	}
	int tsize=tpc->size();
	for(int i=0;i<tsize;i++)
	{
		if((*tpc)[i].m_pProxy0->m_clientObject==static_cast<void*>(tgb))
		{
			gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(*tpc)[i].m_pProxy1->m_clientObject);
			if(tb)
				gmbLocalGlobal_List.push_back(tb);
		}
		else if((*tpc)[i].m_pProxy1->m_clientObject==static_cast<void*>(tgb))
		{
			gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(*tpc)[i].m_pProxy0->m_clientObject);
			if(tb)
				gmbLocalGlobal_List.push_back(tb);
		}
	}
	return gmbLocalGlobal_List.size();

}
GM_export GM_Real GMBULLET_ClearList()
{
	gmbLocalGlobal_List.clear();
	return 0;
}
GM_export GM_Real GMBULLET_GetListBodyID(GM_Real num,GM_Real idtype)
{
	if((size_t)num<unsigned int(gmbLocalGlobal_List.size()))
	{
		if(idtype)
			return gmbLocalGlobal_List[unsigned int(num)]->m_GMLIndex;
		else
			return gmbLocalGlobal_List[unsigned int(num)]->m_GMBIndex;
	}
	return -1;
}
GM_export GM_Real GMBULLET_GetListBodyIDsToStack(GM_Real stackID,GM_Real idtype)
{
    if(idtype)
    {
        for(int i=0;i<gmbLocalGlobal_List.size();i++)
        {
            ds_stack_push(stackID,gmbLocalGlobal_List[i]->m_GMLIndex);
        }
    }
    else
    {
        for(int i=0;i<gmbLocalGlobal_List.size();i++)
        {
            ds_stack_push(stackID,gmbLocalGlobal_List[i]->m_GMBIndex);
        }
    }
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetListBodyIDsToList(GM_Real listID,GM_Real idtype)
{
    if(idtype)
    {
        for(int i=0;i<gmbLocalGlobal_List.size();i++)
        {
            ds_list_add(listID,gmbLocalGlobal_List[i]->m_GMLIndex);
        }
    }
    else
    {
        for(int i=0;i<gmbLocalGlobal_List.size();i++)
        {
            ds_list_add(listID,gmbLocalGlobal_List[i]->m_GMBIndex);
        }
    }
	return ds_list_size(listID);
}
//Collision Pairs
GM_export GM_Real GMBULLET_GetNumNewCollisionPair()
{
	if(!gmbGlobal_myOverPairCall)
		return -1;
	return gmbGlobal_myOverPairCall->newPair.size();
}
GM_export GM_Real GMBULLET_GetNewCollisionPairBodyID(GM_Real num,GM_Real bodyB,GM_Real idtype)
{
	if(!gmbGlobal_myOverPairCall)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_myOverPairCall->newPair.size())<=tnum || tnum<0)
		return -2;
	if(bodyB)
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_myOverPairCall->newPair[tnum].m_pProxy1->m_clientObject));
		if(!tb)
			return -3;
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
	else
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_myOverPairCall->newPair[tnum].m_pProxy0->m_clientObject));
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
}
GM_export GM_Real GMBULLET_GetNumOldCollisionPair()
{
	if(!gmbGlobal_myOverPairCall)
		return -1;
	return gmbGlobal_myOverPairCall->oldPair.size();
}
GM_export GM_Real GMBULLET_GetOldCollisionPairBodyID(GM_Real num,GM_Real bodyB,GM_Real idtype)
{
	if(!gmbGlobal_myOverPairCall)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_myOverPairCall->oldPair.size())<=tnum || tnum<0)
		return -2;
	if(bodyB)
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_myOverPairCall->oldPair[tnum].m_pProxy1->m_clientObject));
		if(!tb)
			return -3;
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
	else
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_myOverPairCall->oldPair[tnum].m_pProxy0->m_clientObject));
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
}

GM_export GM_Real GMBULLET_GetOldCollisionPairToStack(GM_Real stackID,GM_Real idtype)
{
    btBroadphasePairArray* tp;
	tp=&gmbGlobal_myOverPairCall->oldPair;

	if(idtype)
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_stack_push(stackID,tb2->m_GMLIndex);
				ds_stack_push(stackID,tb1->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_stack_push(stackID,tb2->m_GMBIndex);
				ds_stack_push(stackID,tb1->m_GMBIndex);
			}
		}
	}
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetOldCollisionPairToList(GM_Real listID,GM_Real idtype)
{
      btBroadphasePairArray* tp;
	tp=&gmbGlobal_myOverPairCall->oldPair;

	if(idtype)
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_list_add(listID,tb1->m_GMLIndex);
				ds_list_add(listID,tb2->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_list_add(listID,tb1->m_GMBIndex);
				ds_list_add(listID,tb2->m_GMBIndex);
			}
		}
	}
	return ds_list_size(listID);
}
GM_export GM_Real GMBULLET_GetNewCollisionPairToStack(GM_Real stackID,GM_Real idtype)
{
    btBroadphasePairArray* tp;
    tp=&gmbGlobal_myOverPairCall->newPair;

	if(idtype)
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_stack_push(stackID,tb2->m_GMLIndex);
				ds_stack_push(stackID,tb1->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_stack_push(stackID,tb2->m_GMBIndex);
				ds_stack_push(stackID,tb1->m_GMBIndex);
			}
		}
	}
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetNewCollisionPairToList(GM_Real listID,GM_Real idtype)
{
    btBroadphasePairArray* tp;
	tp=&gmbGlobal_myOverPairCall->newPair;

	if(idtype)
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_list_add(listID,tb1->m_GMLIndex);
				ds_list_add(listID,tb2->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0; i<tp->size();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy0->m_clientObject));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)((*tp)[i].m_pProxy1->m_clientObject));
			if(tb2&&tb1)
			{
				ds_list_add(listID,tb1->m_GMBIndex);
				ds_list_add(listID,tb2->m_GMBIndex);
			}
		}
	}
	return ds_list_size(listID);
}
/////////////////////////////////////////////////////////
GM_export GM_Real GMBULLET_GetNumCollisionPair()
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	return gmbGlobal_World->getDispatcher()->getNumManifolds();
}
GM_export GM_Real GMBULLET_GetCollisionPairBodyID(GM_Real num,GM_Real bodyB,GM_Real idtype)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	if(bodyB)
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getBody1()));
		if(!tb)
			return -3;
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
	else
	{
		gmbRigidBody* tb=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getBody0()));
		if(idtype)
			return tb->m_GMLIndex;
		else
			return tb->m_GMBIndex;
	}
}

GM_export GM_Real GMBULLET_GetCollisionPairNumContactPoints(GM_Real num)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	return unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds());
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointX(GM_Real num,GM_Real point,GM_Real PointOnB)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	size_t tpnum=size_t(point);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	if(unsigned int(gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getNumContacts())<=tpnum || tpnum<0)
		return -2;
	if(PointOnB)
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnB().getX();
	else
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnA().getX();
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointY(GM_Real num,GM_Real point,GM_Real PointOnB)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	size_t tpnum=size_t(point);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	if(unsigned int(gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getNumContacts())<=tpnum || tpnum<0)
		return -2;
	if(PointOnB)
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnB().getY();
	else
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnA().getY();
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointZ(GM_Real num,GM_Real point,GM_Real PointOnB)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	size_t tpnum=size_t(point);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	if(unsigned int(gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getNumContacts())<=tpnum || tpnum<0)
		return -2;
	if(PointOnB)
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnB().getZ();
	else
		return gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(tpnum).getPositionWorldOnA().getZ();
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointAverageX(GM_Real num)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	int tmnum=gmbGlobal_World->getDispatcher()->getNumManifolds();
	btVector3 tpoint=btVector3(0,0,0);
	for(int i=0;i<tmnum;i++)
	{
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnA();
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnB();
	}
	tpoint/=2.0f*float(tmnum);
	return tpoint.getX();
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointAverageY(GM_Real num)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	int tmnum=gmbGlobal_World->getDispatcher()->getNumManifolds();
	btVector3 tpoint=btVector3(0,0,0);
	for(int i=0;i<tmnum;i++)
	{
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnA();
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnB();
	}
	tpoint/=2.0f*float(tmnum);
	return tpoint.getY();
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointAverageZ(GM_Real num)
{
	if(!gmbGlobal_IsInitialized)
		return -1;
	size_t tnum=size_t(num);
	if(unsigned int(gmbGlobal_World->getDispatcher()->getNumManifolds())<=tnum || tnum<0)
		return -2;
	int tmnum=gmbGlobal_World->getDispatcher()->getNumManifolds();
	btVector3 tpoint=btVector3(0,0,0);
	for(int i=0;i<tmnum;i++)
	{
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnA();
		tpoint+=gmbGlobal_World->getDispatcher()->getManifoldByIndexInternal(tnum)->getContactPoint(i).getPositionWorldOnB();
	}
	tpoint/=2.0f*float(tmnum);
	return tpoint.getZ();
}

GM_export GM_Real GMBULLET_GetCollisionPairToStack(GM_Real stackID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody1()));
			if(tb1&&tb2)
			{
				ds_stack_push(stackID,tb2->m_GMLIndex);
				ds_stack_push(stackID,tb1->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody1()));
			if(tb1&&tb2)
			{
				ds_stack_push(stackID,tb2->m_GMBIndex);
				ds_stack_push(stackID,tb1->m_GMBIndex);
			}
		}
	}
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetCollisionPairToList(GM_Real listID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody1()));
			if(tb1&&tb2)
			{
				ds_list_add(listID,tb1->m_GMLIndex);
				ds_list_add(listID,tb2->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(td->getManifoldByIndexInternal(i)->getBody1()));
			if(tb1&&tb2)
			{
				ds_list_add(listID,tb1->m_GMBIndex);
				ds_list_add(listID,tb2->m_GMBIndex);
			}
		}
	}
	return ds_list_size(listID);
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointAverageToStack(GM_Real stackID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				btVector3 average_cp(0,0,0);
				for(int j=0;j<tm->getNumContacts();j++)
				{
					average_cp+=(tm->getContactPoint(j).getPositionWorldOnA()+tm->getContactPoint(j).getPositionWorldOnB());
				}
				average_cp/=(2.0f*float(tm->getNumContacts()));
				ds_stack_push(stackID,average_cp.getZ());
				ds_stack_push(stackID,average_cp.getY());
				ds_stack_push(stackID,average_cp.getX());
				ds_stack_push(stackID,tb2->m_GMLIndex);
				ds_stack_push(stackID,tb1->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				btVector3 average_cp(0,0,0);
				for(int j=0;j<tm->getNumContacts();j++)
				{
					average_cp+=(tm->getContactPoint(j).getPositionWorldOnA()+tm->getContactPoint(j).getPositionWorldOnB());
				}
				average_cp/=(2.0f*float(tm->getNumContacts()));
				ds_stack_push(stackID,average_cp.getZ());
				ds_stack_push(stackID,average_cp.getY());
				ds_stack_push(stackID,average_cp.getX());
				ds_stack_push(stackID,tb2->m_GMBIndex);
				ds_stack_push(stackID,tb1->m_GMBIndex);
			}
		}
	}
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointsToStack(GM_Real stackID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				for(int j=0;j<tm->getNumContacts();j++)
				{
					btVector3 tcp=tm->getContactPoint(j).getPositionWorldOnB();
					ds_stack_push(stackID,tcp.getZ());
					ds_stack_push(stackID,tcp.getY());
					ds_stack_push(stackID,tcp.getX());
					tcp=tm->getContactPoint(j).getPositionWorldOnA();
					ds_stack_push(stackID,tcp.getZ());
					ds_stack_push(stackID,tcp.getY());
					ds_stack_push(stackID,tcp.getX());
				}
				ds_stack_push(stackID,tm->getNumContacts());
				ds_stack_push(stackID,tb2->m_GMLIndex);
				ds_stack_push(stackID,tb1->m_GMLIndex);
			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				for(int j=0;j<tm->getNumContacts();j++)
				{
					btVector3 tcp=tm->getContactPoint(j).getPositionWorldOnB();
					ds_stack_push(stackID,tcp.getZ());
					ds_stack_push(stackID,tcp.getY());
					ds_stack_push(stackID,tcp.getX());
					tcp=tm->getContactPoint(j).getPositionWorldOnA();
					ds_stack_push(stackID,tcp.getZ());
					ds_stack_push(stackID,tcp.getY());
					ds_stack_push(stackID,tcp.getX());
				}
				ds_stack_push(stackID,tm->getNumContacts());
				ds_stack_push(stackID,tb2->m_GMBIndex);
				ds_stack_push(stackID,tb1->m_GMBIndex);
			}
		}
	}
	return ds_stack_size(stackID);
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointAverageToList(GM_Real listID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				btVector3 average_cp(0,0,0);
				for(int j=0;j<tm->getNumContacts();j++)
				{
					average_cp+=(tm->getContactPoint(j).getPositionWorldOnA()+tm->getContactPoint(j).getPositionWorldOnB());
				}
				average_cp/=(2.0f*float(tm->getNumContacts()));
				ds_list_add(listID,tb1->m_GMLIndex);
				ds_list_add(listID,tb2->m_GMLIndex);
				ds_list_add(listID,average_cp.getX());
				ds_list_add(listID,average_cp.getY());
				ds_list_add(listID,average_cp.getZ());
			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				btVector3 average_cp(0,0,0);
				for(int j=0;j<tm->getNumContacts();j++)
				{
					average_cp+=(tm->getContactPoint(j).getPositionWorldOnA()+tm->getContactPoint(j).getPositionWorldOnB());
				}
				average_cp/=(2.0f*float(tm->getNumContacts()));
				ds_list_add(listID,tb1->m_GMBIndex);
				ds_list_add(listID,tb2->m_GMBIndex);
				ds_list_add(listID,average_cp.getX());
				ds_list_add(listID,average_cp.getY());
				ds_list_add(listID,average_cp.getZ());
			}
		}
	}
	return ds_list_size(listID);
}
GM_export GM_Real GMBULLET_GetCollisionPairContactPointsToList(GM_Real listID,GM_Real idtype)
{
	btDispatcher* td=gmbGlobal_World->getDispatcher();
	if(idtype)
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				ds_list_add(listID,tb1->m_GMLIndex);
				ds_list_add(listID,tb2->m_GMLIndex);
				ds_list_add(listID,tm->getNumContacts());
				for(int j=0;j<tm->getNumContacts();j++)
				{
					btVector3 tcp=tm->getContactPoint(j).getPositionWorldOnA();
					ds_list_add(listID,tcp.getX());
					ds_list_add(listID,tcp.getY());
					ds_list_add(listID,tcp.getZ());
					tcp=tm->getContactPoint(j).getPositionWorldOnB();
					ds_list_add(listID,tcp.getX());
					ds_list_add(listID,tcp.getY());
					ds_list_add(listID,tcp.getZ());
				}

			}
		}
	}
	else
	{
		for(int i=0;i<td->getNumManifolds();i++)
		{
			btPersistentManifold* tm=td->getManifoldByIndexInternal(i);
			gmbRigidBody* tb1=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody0()));
			gmbRigidBody* tb2=dynamic_cast<gmbRigidBody*>((btCollisionObject*)(tm->getBody1()));
			if(tb1&&tb2)
			{
				ds_list_add(listID,tb1->m_GMBIndex);
				ds_list_add(listID,tb2->m_GMBIndex);
				ds_list_add(listID,tm->getNumContacts());
				for(int j=0;j<tm->getNumContacts();j++)
				{
					btVector3 tcp=tm->getContactPoint(j).getPositionWorldOnA();
					ds_list_add(listID,tcp.getX());
					ds_list_add(listID,tcp.getY());
					ds_list_add(listID,tcp.getZ());
					tcp=tm->getContactPoint(j).getPositionWorldOnB();
					ds_list_add(listID,tcp.getX());
					ds_list_add(listID,tcp.getY());
					ds_list_add(listID,tcp.getZ());
				}

			}
		}
	}
	return ds_list_size(listID);
}

//-----------------------------------