// Shader Extension
// Version: 1.4
// Updated: 22/Dec/2010
// by LSnK

/*
  Changelog

    1.4: Fixed d3d_vs_create(), it was ignoring constants set by the 'def' instruction.
    1.4: Changed VS configs so they work like the others.  No overwriting without explicit definition.

    1.3: Added z-bias control.  Prevents z-fighting between coplanar polygons; useful for shadows and decals.
    1.3: Added extended primitive functions which can draw up to 8192 vertices at once with position, normal, colour, specular, eight textures and eight sets of texture coordinates.
    1.3: Added vertex shader 1.1 support and related functions.  Due to GM's nefarious machinations, their use is limited to the the extended primitive functions.
    1.3: Added helper functions which directly set shader constants to GM-format colours.
    1.3: Changed configs: PS/VS/tex configs are now independent.
    1.3: Changed d3d_set_tex_all: now sets stage 0 too.  The extended primitive functions use it.
    1.3: Demo has been updated with a water post-processing effect.  Press P.
    
    1.2: Added d3d_dev_get_ps_version, checks pixel shader support.
    1.2: Added d3d_set_tex_border, sets colour and alpha for tex_wrap_border mode.
    1.2: d3d_set_tex* functions now accept -1 as tex parameter. This unbinds the texture from the stage making it safe to delete the associated resource in GM.
    1.2: Fixed the extension's fillmode_ constants. They were named incorrectly.
    1.2: Fixed d3d_set_tex_all in the DLL version. It was defined with the wrong number of arguments.
    1.2: Fixed d3d_set_fog_color setting the wrong colours. D3D's colour format is RGB, not BGR.

    1.1: First release.
*/



#define GMAPI_USE_D3D
#include <string>
#include <vector>
#include <fstream>
#include <math.h>
#include <gmapi.h>
#include <d3dx8.h>
#include "fprot.h"


#define gmxd extern "C" __declspec (dllexport) double
#define gmxs extern "C" __declspec (dllexport) char*
#define mod %
#define and &&
#define or ||
#define gtrue 1.0     // Success
#define gfalse 0.0    // Failure
#define gerror -1.0   // User input error
#define uint unsigned int
#define usint unsigned short int
#define pi 3.1415926535897932384626433832795
#define crlf "\r\n"
#define epsilon 0.00001
#define var gm::CGMVariable
#define complain(x) gm::show_error(x,false); return gerror;
#define ohshit 0xBABEBEEF

// Define away a bunch of gibberish...
#define return_gmxs(str)    str_ret = str;  return (char*) str_ret.c_str();
#define dword               DWORD                               // Capslock is autopilot for cool!
#define d3dvar(x)           *((DWORD*)&x)                       // Float as dword pointer for D3D's more shitastic functions.
#define d3dcheck(f)         return (double) (D3D_OK == (f))     // Returns status immediately.
#define d3dfail(v,f)        if (D3D_OK != (f)) { return v; }    // Returns status only on fail.
#define d3drs(state,value)  d3ddev->SetRenderState(state,value) // Guess.
#define d3dcrs(state,value) d3dcheck(d3drs(state,value))        // Set RS and return status immediately.
#define d3dtex              IDirect3DTexture8

// Buffer parameters
#define fvf_default  ( D3DFVF_XYZ | D3DFVF_NORMAL  | D3DFVF_DIFFUSE  | D3DFVF_TEX1 ) // GM's FVF for D3D.
#define fvf_ext      ( D3DFVF_XYZ | D3DFVF_NORMAL  | D3DFVF_DIFFUSE  | D3DFVF_SPECULAR | D3DFVF_TEX8 )  // M-M-M-MEGA KILL
#define vb_count     8192 // Max verts per prim; GM's limit of 1024 is excessively conservative.
#define vb_bytes     vb_count*sizeof(vert_ext)  // 768KB


// ============================================================================
// GMAPI setup
// ============================================================================


static gm::CGMAPI* gmapi;

bool WINAPI DllMain( HINSTANCE aModuleHandle, int aReason, int aReserved ) 
{
  switch ( aReason ) 
  {
    case DLL_PROCESS_ATTACH: 
    {
       long uint result = 0;
       gmapi = gm::CGMAPI::Create( &result );
    } break;

    case DLL_PROCESS_DETACH:
    {
       gmapi->Destroy();
    } break;
  }

  return true;
}




// ============================================================================
// Internal functions / structures
// ============================================================================
using namespace std;

const static double degtorad_mul = pi/180.0;
const static double radtodeg_mul = 180.0/pi;
const static double pi2          = pi*2.0;



// Handy functions
inline double round     ( double x )
{
    return ((x<0.0) ? ceil(x-0.5) : floor(x+0.5));
}
inline double frac      ( double x )
{
    return fmod(x,1.0);
}
inline bool   eps       ( double a, double b=0.0 )
{
 return (fabs(a-b) < epsilon);
}
inline bool   eps2      ( double a, double b, double eps )
{
 return (fabs(a-b) < eps);
}
inline double slope     ( double x1, double y1, double x2, double y2 )
{
    // Slope of a line
    double dx;
    dx = x2-x1;

    if (eps(dx))
         {return 0.0;}
    else {return (y2-y1) / dx;}
}
inline double lerp      ( double vala, double valb, double x )
{
    // Linear interpolation.
    return vala + ((valb-vala)*x);
}
inline double cerp      ( double vala, double valb, double x )
{
    // Cosine interpolation.
    double am2 = (1 - cos(x*pi)) * 0.5;
    return vala + ((valb-vala)*am2);
}
inline double curp      ( double a, double b, double c, double d, double x )
{
    // Cubic interpolation.  0-1 == B-C.  A and D are the datapoints before and after those.
    double p,q;
    p = (d-c) - (a-b);
    q = (a-b) - p;

    return (p*x*x*x) + (q*x*x) + ((c-a)*x) + b;
}
inline double unlerp    ( double val, double minval, double maxval )
{
    // Returns the value normalised into the range 0-1.
    return (val-minval) / (maxval-minval);
}
inline double clamp     ( double val, double minval, double maxval )
{
    if (val < minval)
    { return minval; }
    else
    if (val > maxval)
    { return maxval; }

    return val;
}
inline double snap_low  ( double x, double cellw )
{
	return x - fmod(x,cellw);
}
inline uint   snap_low  ( uint   x, uint   cellw )
{
    return x - (x mod cellw);
}
inline double snap_high ( double x, double cellw )
{
	return (x - fmod(x,cellw)) + cellw;
}
inline double snap_near ( double x, double cellw )
{
	return round(x/cellw)*cellw;
}
inline void   tr_rot    ( double &x, double &y, double dir )
{
    // Rotate points around 0,0.
	double rad,s,c;
	rad = dir*degtorad_mul;
	s   = -sin(rad);
	c   = cos(rad);
	x   = (x*c) - (y*s);
	y   = (x*s) + (y*c);
}
inline void   tr_move   ( double &x, double &y, double len, double dir )
{
    // Move point with polar coordinates.  Same as lengthdir.
	double rad;
	rad = dir*degtorad_mul;
    x  +=  cos(rad)*len;
	y  += -sin(rad)*len;
}
inline void   tr_scale  ( double &x, double &y, double scale )
{
	x *= scale;
	y *= scale;
}
inline int    col_red   ( int col )
{
	return (col mod 256);
}
inline int    col_green ( int col )
{
	return ((col >> 8) mod 256);
}
inline int    col_blue  ( int col )
{
	return (col >> 16);
}
inline int    col_make  ( int r, int g, int b )
{
	return (b<<16) + (g<<8) + r;
}
int           col_lerp  ( int cola, int colb, double a )
{
    // Lerp between colours.  Same as merge_color in GML.
	int r,g,b;
	r = (int) lerp( col_red  (cola), col_red  (colb), a );
	g = (int) lerp( col_green(cola), col_green(colb), a );
	b = (int) lerp( col_blue (cola), col_blue (colb), a );

	return col_make(r,g,b);
}
uint          col_d3d   ( int gmcol, double gmalpha )
{
    // For some reason GM uses BGR instead of D3D's normal RGB format.
    uint r,g,b,a;
    r = col_red   ( gmcol );
    g = col_green ( gmcol );
    b = col_blue  ( gmcol );
    a = uint(clamp(gmalpha,0.0,1.0)*255.0);
   
    return (a<<24) + (r<<16) + (g<<8) + b;
}

bool file_read  (  char* file_in,  char* &target, uint &size  )
{
    // Load file into RAM.
    // You need to declare the target pointer and size vars first.
    // Returns whether it succeeded.  If it fails, there's nothing to clean up.
    // If it works you need to delete[] the buffer when you're done.
    using namespace std;
    fstream f;

    f.open( file_in, ios::in | ios::binary );
    if (f.is_open())
    {
         f.seekg( 0, ios::end );
         size  = (uint) f.tellg();   
         
         target = new (nothrow) char [size];
         if (target != nullptr)
         {
             f.seekg( 0, ios::beg );
             f.read ( target, size );
             f.close();
             return true;
         }
         else
         {
             f.close();
             return false;
         }
         
    } 
    else 
    {
         return false;
    }
}
bool file_write (  char* file_out, void*  data,  uint size,  bool append=false  )
{
    // Write from RAM to file.  Returns whether it succeeded.
    using namespace std;
    fstream f;

    if (append) { f.open( file_out, ios::out | ios::binary | ios::app ); }
    else        { f.open( file_out, ios::out | ios::binary );            }

    if (f.is_open())
    {
     f.write( (char*) data, size );
     f.close();
    }
    else {return false;}

    return true;
}
bool file_write (  char* file_out, string data,  uint size,  bool append=false  )
{
    return file_write( file_out, data.data(), min(size,data.length()), append );
}





// ============================================================================
// External functions 
// ============================================================================

using namespace std;

struct ps_const
{
 // Pixel shader constant.
 float r;
 float g;
 float b;
 float a;
};
struct ps_conf
{
    // User-defined constant register states
    bool     set [8];
    ps_const c   [8];  // Values
};
struct vs_const
{
 // Vertex shader constant.  
 float x;
 float y;
 float z;
 float w;
};
struct vs_conf
{
    // User-defined constant register states
    vs_const c  [96];     // Hardware suports more, but GM is software-only.
    bool     set[96];
};
struct tex_conf
{
    // User-defined texture stage states
    bool     set   [8];  // Is this tex stage set?
    d3dtex*  tex   [8];  // Texture pointer
    dword    in    [8];  // Texture interpolation mode
    dword    xwrap [8];  // Texture wrapping modes
    dword    ywrap [8];  // 
};
struct vert_default
{
    // GM's default D3D type, 36 bytes.
    float x,y,z;
    float nx,ny,nz;
    dword c;
    float uv [2];
};
struct vert_ext
{
    // Extended type, 96 bytes.
    float x,y,z;     // Position
    float nx,ny,nz;  // Normal
    dword c;         // Diffuse col,  PS v0
    dword s;         // Specular col, PS v1
    float uv[16];    // Texcoords,    PS t0-t5.  6+7 are not accessible to pixel shaders.
};

static IDirect3DDevice8*       d3ddev;  // D3D device pointer
static IDirect3D8*	           d3dint;  // D3D interface pointer
static D3DCAPS8                d3dcaps; // GPU capability struct
static D3DADAPTER_IDENTIFIER8  d3daid;  // GPU identification struct

static LPDIRECT3DVERTEXBUFFER8 vbuff_d3d;            // Pointer to D3D vertex buffer
static vert_ext                vbuff_int [vb_count]; // Internal vertex buffer
static uint                    vbuff_c;              // Internal counter
static D3DPRIMITIVETYPE        vbuff_prim;           // Primitive to draw
static bool                    vbuff_usevs;          // Use vertex shader?
static bool                    vbuff_autoinc;        // Automatic increment?

static vector<ps_conf>         conf_vec_ps;          // Dynamic arrays for configs
static vector<vs_conf>         conf_vec_vs;          // 
static vector<tex_conf>        conf_vec_tex;         // 

static string                  str_ret = "BABEBEEF"; // Used to return strings by macro.


// Initialisation
gmxd init ()
{
    // Initialises device pointer, GPU information, buffers, etc.
    d3ddev = gmapi->GetDirect3DDevice();
    d3dint = gmapi->GetDirect3DInterface();

    d3dint->GetAdapterIdentifier( D3DADAPTER_DEFAULT, D3DENUM_NO_WHQL_LEVEL, &d3daid );
    d3ddev->GetDeviceCaps( &d3dcaps );

    d3ddev->CreateVertexBuffer( vb_bytes, D3DUSAGE_WRITEONLY, fvf_ext, D3DPOOL_MANAGED, &vbuff_d3d );

    return gtrue;
}

// Information
gmxs d3d_dev_get_name           ()
{
	// GPU name.
    return_gmxs( d3daid.Description );
}
gmxd d3d_dev_get_point_max_size ()
{
	// Maximum size of point primitives.
    return (double) d3dcaps.MaxPointSize;
}
gmxd d3d_dev_get_ps_version     ()
{
    // GPU pixel shader version.  10 to 14.  Most modern GPUs support higher versions but they're not reported here.
    uint v = (uint) d3dcaps.PixelShaderVersion;
    return (double) ((((v>>8)&0xFF)*10)+v&0xFF);
}
gmxd d3d_dev_get_tex_max_width  ()
{
    // Maximum texture width.  Applies to all graphical resources.
    return (double) d3dcaps.MaxTextureWidth;
}
gmxd d3d_dev_get_tex_max_height ()
{
    // Height.
    return (double) d3dcaps.MaxTextureHeight;
}
gmxd d3d_dev_get_tex_max_stages ()
{
    // Maximum simultaneous textures.  Limits how many texture stages you can use.
    return (double) d3dcaps.MaxSimultaneousTextures;
}
gmxd d3d_dev_get_tex_mem        ()
{
    // Free texture memory in bytes.  Approximate.  This ISN'T the VRAM size.
    return (double) d3ddev->GetAvailableTextureMem();
}

// Pixel shaders
gmxd d3d_ps_create        ( char*  src_asm ) 
{
    // Assembles and creates pixel shader.  Returns handle.
    string       str = src_asm;
    string       err;
    dword        shader;
    LPD3DXBUFFER psc;
    LPD3DXBUFFER errors;

    if (d3dcaps.PixelShaderVersion < D3DPS_VERSION(1,4))
    {
     err = "PS 1.4 unsupported by GPU.";
     complain( err.c_str() );
     return gfalse;
    }

    if ( D3D_OK != D3DXAssembleShader( (LPCVOID) str.c_str(), str.length(), 0, nullptr, &psc, &errors ) )
    {
     err.append( crlf );
     err.append( "Shader assembly error:" );
     err.append( crlf );
     err.append( crlf );
     err.append( (char*) errors->GetBufferPointer() );
     err.append( crlf );
     err.append( crlf );
     err.append( str );
     complain( err.c_str() );
     return gerror;
    }

    if ( D3D_OK != d3ddev->CreatePixelShader( (dword*) psc->GetBufferPointer(), &shader ) )
    {
     err = "Shader creation failed.  This should never happen.";
     complain( err.c_str() );
     return gfalse;
    }

    return (double) shader;
}
gmxd d3d_ps_destroy       ( double shader  )
{
    // Free shader.
    d3dcheck( d3ddev->DeletePixelShader( (dword) shader ) );
}
gmxd d3d_set_ps           ( double shader )
{
    // Set active pixel shader, or -1 to disable.
    if (shader < 0)
         { d3dcheck( d3ddev->SetPixelShader( NULL           ) ); }
    else { d3dcheck( d3ddev->SetPixelShader( (dword) shader ) ); }
}
gmxd d3d_set_ps_ext       ( double shader, double conf )
{
    // Set active pixel shader and configuration.
    bool a,b;
    a = ( d3d_set_ps      ( shader ) > 0.0);
    b = ( d3d_set_ps_conf ( conf   ) > 0.0);

    if (a and b)
         { return gtrue;  }
    else { return gerror; }
}
gmxd d3d_set_ps_const     ( double constant, double r, double g, double b, double a )
{
    // Set pixel shader constant register.  There are 8 registers indexed 0-7 each containing 4 floating point values between -1 and 1.  This is the same as using "def" in the shader.
    ps_const cx;
    cx.r  = (float) clamp(r,-1.0,1.0);
    cx.g  = (float) clamp(g,-1.0,1.0);
    cx.b  = (float) clamp(b,-1.0,1.0);
    cx.a  = (float) clamp(a,-1.0,1.0);

    d3dcheck( d3ddev->SetPixelShaderConstant( (dword) constant, &cx, 1 ) );
}
gmxd d3d_set_ps_const_col ( double constant, double col, double alpha )
{
    // Set PS constant as colour.  Handy shortcut.
    int      c = (int) col;
    ps_const cx;
    cx.r  = (float(col_red  (c))/255.0f);
    cx.g  = (float(col_green(c))/255.0f);
    cx.b  = (float(col_blue (c))/255.0f);
    cx.a  = (float) clamp(alpha,0.0,1.0);

    d3dcheck( d3ddev->SetPixelShaderConstant( (dword) constant, (void*) &cx, 1 ));
}
gmxd d3d_set_ps_conf      ( double conf )
{
    // Set PS constant registers from a predefined configuration.  Constants not set in the config aren't overwritten.
    if (conf > conf_vec_ps.size()-1)
    { return gerror; }

    uint     vpos = (uint) floor(conf);
    ps_conf* px   = &(conf_vec_ps[vpos]);

    for (uint i=0; i<8; i++)
    {
        if (px->set[i])
        { d3ddev->SetPixelShaderConstant( i, &(px->c[i]), 1 ); }
    }
        
    return gtrue;
}

// Vertex shaders
gmxd d3d_vs_create           ( char*  src_asm ) 
{
    // Assembles and creates vertex shader.  Returns handle.
    // Whoever wrote the D3D API should be punched in the balls.  Just for the record.
    string       str = src_asm;
    string       err;
    DWORD        shader;
    LPD3DXBUFFER vsc;
    LPD3DXBUFFER constants;
    LPD3DXBUFFER errors;

    if ( D3D_OK != D3DXAssembleShader( (LPCVOID) str.c_str(), str.length(), 0, &constants, &vsc, &errors ) )
    {
     err.append( crlf );
     err.append( "Shader assembly error:" );
     err.append( crlf );
     err.append( crlf );
     err.append( (char*) errors->GetBufferPointer() );
     err.append( crlf );
     err.append( crlf );
     err.append( str );
     complain( err.c_str() );
     return gerror;
    }

    
    const uint o      = (96*5); // 96 sets of [d3dvsd_const + 4 floats]
    dword      vsdec  [o+32];

    SecureZeroMemory( (PVOID) vsdec,  sizeof(vsdec)  );
    memcpy( vsdec, constants->GetBufferPointer(), constants->GetBufferSize() );

    vsdec[o]    = D3DVSD_STREAM( 0 );                                 // VSR     V-out    PSR
    vsdec[o+1]  = D3DVSD_REG( D3DVSDE_POSITION,  D3DVSDT_FLOAT3   );  // v0  ->  oPos
    vsdec[o+2]  = D3DVSD_REG( D3DVSDE_NORMAL,    D3DVSDT_FLOAT3   );  // v3
    vsdec[o+3]  = D3DVSD_REG( D3DVSDE_DIFFUSE,   D3DVSDT_D3DCOLOR );  // v5  ->  oD0   -> v0
    vsdec[o+4]  = D3DVSD_REG( D3DVSDE_SPECULAR,  D3DVSDT_D3DCOLOR );  // v6  ->  oD1   -> v1
    vsdec[o+5]  = D3DVSD_REG( D3DVSDE_TEXCOORD0, D3DVSDT_FLOAT2   );  // v7  ->  oT0   -> t0
    vsdec[o+6]  = D3DVSD_REG( D3DVSDE_TEXCOORD1, D3DVSDT_FLOAT2   );  // v8  ->  oT1   -> t1
    vsdec[o+7]  = D3DVSD_REG( D3DVSDE_TEXCOORD2, D3DVSDT_FLOAT2   );  // v9  ->  oT2   -> t2
    vsdec[o+8]  = D3DVSD_REG( D3DVSDE_TEXCOORD3, D3DVSDT_FLOAT2   );  // v10 ->  oT3   -> t3
    vsdec[o+9]  = D3DVSD_REG( D3DVSDE_TEXCOORD4, D3DVSDT_FLOAT2   );  // v11 ->  oT4   -> t4
    vsdec[o+10] = D3DVSD_REG( D3DVSDE_TEXCOORD5, D3DVSDT_FLOAT2   );  // v12 ->  oT5   -> t5
    vsdec[o+11] = D3DVSD_REG( D3DVSDE_TEXCOORD6, D3DVSDT_FLOAT2   );  // v13 ->  oT6
    vsdec[o+12] = D3DVSD_REG( D3DVSDE_TEXCOORD7, D3DVSDT_FLOAT2   );  // v14 ->  oT7
    vsdec[o+13] = D3DVSD_END();

    if ( D3D_OK != d3ddev->CreateVertexShader( vsdec, (dword*) vsc->GetBufferPointer(), &shader, D3DUSAGE_SOFTWAREPROCESSING ) )
    {
     err.append( crlf );
     err.append( "Vertex shader creation failed." );
     err.append( crlf );
     err.append( crlf );
     err.append( str );
     complain( err.c_str() );
     return gerror;
    }

    return (double) shader;
}
gmxd d3d_vs_destroy          ( double shader  )
{
    // Free vertex shader.
    d3dcheck( d3ddev->DeleteVertexShader( (DWORD) shader ) );
}
gmxd d3d_set_vs              ( double shader  )
{
    // Set current vertex shader or -1 for none.  GM's normal drawing functions are not affected.
    if (shader < 0)
         { 
           vbuff_usevs = false;
           d3dcheck( d3ddev->SetVertexShader( fvf_ext        ) );
         }
    else { 
           vbuff_usevs = true;
           d3dcheck( d3ddev->SetVertexShader( (dword) shader ) );
         }
}
gmxd d3d_set_vs_const        ( double constant, double x, double y, double z, double w )
{
    // Set VS constant register.  There are 96 4-component registers available.
    vs_const vx;
    vx.x  = (float) x;
    vx.y  = (float) y;
    vx.z  = (float) z;
    vx.w  = (float) w;

    d3dcheck( d3ddev->SetVertexShaderConstant( (dword) clamp(constant,0.0,95.0), &vx, 1 ));
}
gmxd d3d_set_vs_const_col    ( double constant, double col, double alpha )
{
    // Set VS constant as colour.  Handy shortcut.
    int      c = (int) col;
    vs_const vx;
    vx.x  = (float(col_red  (c))/255.0f);
    vx.y  = (float(col_green(c))/255.0f);
    vx.z  = (float(col_blue (c))/255.0f);
    vx.w  = (float) clamp(alpha,0.0,1.0);

    d3dcheck( d3ddev->SetVertexShaderConstant( (dword) clamp(constant,0.0,95.0), &vx, 1 ));
}
gmxd d3d_set_vs_const_matrix ( double constant )
{
    // Sets four constants as the transposed world*view*projection matrix.  You can then use [m4x4 oPos,v0,cn] in the shader to transform the vertices in keeping with GM's normal behaviour.   Ex: 0 would set c0,c1,c2,c3; 4 would set c4,c5,c6,c7.
    D3DXMATRIX world, proj, view, out;
    d3ddev->GetTransform( D3DTS_WORLD,      &world );
    d3ddev->GetTransform( D3DTS_VIEW,       &view  );
    d3ddev->GetTransform( D3DTS_PROJECTION, &proj  );
           
    D3DXMatrixTranspose( &out, &(world * view * proj) );

    d3dcheck( d3ddev->SetVertexShaderConstant( (dword) clamp(constant,0.0,92.0), out, 4 ));
}
gmxd d3d_set_vs_conf         ( double conf )
{
    // Set VS constant registers from a predefined configuration.
    if (conf > conf_vec_vs.size()-1)
    { return gerror; }

    uint     vpos = (uint) floor(conf);
    vs_conf* vx   = &(conf_vec_vs[vpos]);

    for (uint i=0; i<96; i++)
    {
        if (vx->set[i])
        { d3ddev->SetVertexShaderConstant( i, &(vx->c[i]), 1 ); }
    }

    return gtrue;
};

// Texture stages
gmxd d3d_set_tex        ( double stage, double tex )
{
    // Set texture for this texture stage or -1 for none.  Read the texture in a shader by using (texld r<stage>,t0).  GM uses 0 for drawing so don't fiddle with it.
    uint s = (uint) stage;

    if  (s < d3dcaps.MaxSimultaneousTextures)
    {
        if (tex < 0.0)
        {
            d3dcheck( d3ddev->SetTexture( s, NULL ) );
        }
        else
        {
            d3dtex* t = gmapi->GetDirect3DTexture( (int)tex );

            if (t == nullptr)
            { return gerror; }

            d3dcheck( d3ddev->SetTexture(s,t) );
        }
    }
    else { return gerror; }

    return gtrue;
}
gmxd d3d_set_tex_all    ( double tex )
{
    // Set all available texture stages.  -1 for no texture.
    for (uint i=0; i<d3dcaps.MaxSimultaneousTextures; i++)
    {
        if (!d3d_set_tex( i, tex ))
        { return gerror; }
    }

    return gtrue;
}
gmxd d3d_set_tex_int    ( double stage, double mode )
{
    // Set texture stage interpolation mode.  Use tex_int_ constant.  Defaults to nearest, which is usually undesirable.  Stage 0 is also controlled by texture_set_interpolation in GM.
    dword s = (dword) stage;

    if ( stage >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }

    if (( D3D_OK == d3ddev->SetTextureStageState( s, D3DTSS_MAGFILTER, (dword) mode ) )
    and ( D3D_OK == d3ddev->SetTextureStageState( s, D3DTSS_MINFILTER, (dword) mode ) ))
    { return gtrue; }

    return gfalse; 
}
gmxd d3d_set_tex_wrap   ( double stage, double xmode, double ymode )
{
    // Set texture stage wrapping mode.  Use tex_wrap_ constant.
    dword s = (dword) stage;

    if ( stage >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }

    if (( D3D_OK == d3ddev->SetTextureStageState( s, D3DTSS_ADDRESSU, (dword) xmode ) )
    and ( D3D_OK == d3ddev->SetTextureStageState( s, D3DTSS_ADDRESSV, (dword) ymode ) ))
    { return gtrue; }

    return gfalse; 
}
gmxd d3d_set_tex_border ( double stage, double col, double alpha )
{
    // Set colour to use with tex_wrap_border.
    dword s = (dword) stage;

    if ( s >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }
    
    d3dcheck( d3ddev->SetTextureStageState( s, D3DTSS_BORDERCOLOR, col_d3d((int)col,alpha) ) );
}
gmxd d3d_set_tex_aniso  ( double stage, double anisotropy )
{
    // Set anisotropic filtering level for tex_int_anisotropic.  Values: 1,2,4,8,16.  1=none.  Defaults to 1.
    dword s = (dword) stage;

    if ( s >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }
    
    d3dcheck( d3ddev->SetTextureStageState( s, D3DTSS_MAXANISOTROPY, (dword) min(anisotropy,d3dcaps.MaxAnisotropy) ) );
}
gmxd d3d_set_tex_mip    ( double stage, double mode )
{
    // Set mipmap filtering mode.  tex_int_nearest or tex_int_bilinear.  The latter when combined with tex_int_bilinear on the texture itself results in trilinear filtering.
    dword s = (dword) stage;

    if ( s >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }
    
    d3dcheck( d3ddev->SetTextureStageState( s, D3DTSS_MIPFILTER, (dword) mode ) );
}
gmxd d3d_set_tex_conf   ( double conf )
{
    // Set texture stage state from a predefined configuration.
    if (conf > conf_vec_tex.size()-1)
    {return gerror;}

    uint vpos  = (uint) floor(conf);

    for (uint i=0; i<min(8,d3dcaps.MaxSimultaneousTextures); i++)
    {
        if (conf_vec_tex[vpos].set[i])
        {
            d3ddev->SetTexture( i, conf_vec_tex[vpos].tex[i]   );
            d3d_set_tex_int   ( i, conf_vec_tex[vpos].in[i]    );
            d3d_set_tex_wrap  ( i, conf_vec_tex[vpos].xwrap[i], 
                                   conf_vec_tex[vpos].ywrap[i] );
        }
    }
    return gtrue;
}

// Configurations
gmxd d3d_conf_ps_create  ()
{
    // Creates new pixel shader configuration and returns its index.
    ps_conf conf;

    SecureZeroMemory( (PVOID) &conf, sizeof(ps_conf) );

    conf_vec_ps.push_back( conf );
    return (double) conf_vec_ps.size()-1;
}
gmxd d3d_conf_ps_set     ( double conf, double constant, double r, double g, double b, double a )
{
    // Defines PS constant configuration.
    if (conf > conf_vec_ps.size()-1)
    { return gerror; }

    uint vpos = (uint) floor(conf);
    uint cpos = (uint) floor(clamp( constant, 0.0, 7.0 ));

    conf_vec_ps[vpos].c[cpos].r  = (float) clamp(r,-1.0,1.0);
    conf_vec_ps[vpos].c[cpos].g  = (float) clamp(g,-1.0,1.0);
    conf_vec_ps[vpos].c[cpos].b  = (float) clamp(b,-1.0,1.0);
    conf_vec_ps[vpos].c[cpos].a  = (float) clamp(a,-1.0,1.0);
    conf_vec_ps[vpos].set[cpos]  = true;

    return gtrue;
}
gmxd d3d_conf_vs_create  ()
{
    // Creates new vertex shader configuration and returns its index.
    vs_conf conf;

    SecureZeroMemory( (PVOID) &conf, sizeof(vs_conf) );

    conf_vec_vs.push_back( conf );
    return (double) conf_vec_vs.size()-1;
}
gmxd d3d_conf_vs_set     ( double conf, double constant, double x, double y, double z, double w )
{
    // Defines VS constant configuration.
    if (conf > conf_vec_vs.size()-1)
    { return gerror; }

    uint vpos = (uint) floor(conf);
    uint cpos = (uint) floor(clamp( constant, 0.0, 255.0 ));

    conf_vec_vs[vpos].c  [cpos].x  = (float) x;
    conf_vec_vs[vpos].c  [cpos].y  = (float) y;
    conf_vec_vs[vpos].c  [cpos].z  = (float) z;
    conf_vec_vs[vpos].c  [cpos].w  = (float) w;
    conf_vec_vs[vpos].set[cpos]    = true;

    return gtrue;
}
gmxd d3d_conf_tex_create ()
{
    // Creates new texture stage configuration and returns its index.
    tex_conf conf;

    SecureZeroMemory( (PVOID) &conf, sizeof(tex_conf) );

    conf_vec_tex.push_back( conf );
    return (double) conf_vec_tex.size()-1;
}
gmxd d3d_conf_tex_set    ( double conf, double stage, double tex, double interp, double xmode, double ymode )
{
    if (conf > conf_vec_tex.size()-1)
    {return gerror;}

    uint vpos = (uint) floor(conf);
    uint s    = (uint) stage;

    if (s >= d3dcaps.MaxSimultaneousTextures )
    { return gerror; }

    d3dtex* t = gmapi->GetDirect3DTexture((int)tex);

    if (t != nullptr)
    {
        conf_vec_tex[vpos].tex  [s] = t;
        conf_vec_tex[vpos].in   [s] = (dword) interp;
        conf_vec_tex[vpos].xwrap[s] = (dword) xmode;
        conf_vec_tex[vpos].ywrap[s] = (dword) ymode;
        conf_vec_tex[vpos].set  [s] = true;
        return gtrue;
    }
    else 
    {   return gerror;   }
}

// Fog
gmxd d3d_set_fog_state   ( double state    )
{
    // Turn fog on or off.
    d3dcrs( D3DRS_FOGENABLE, (state > 0.0) );
}
gmxd d3d_set_fog_type    ( double fog_type )
{
    // Set the type of fog.  Use fog_type_ constants.  Linear by default.
    d3dcrs( D3DRS_FOGTABLEMODE, (dword) fog_type );
}
gmxd d3d_set_fog_density ( double density  )
{
    // Controls density of exponential fog. 0-1.
    float d = (float) clamp( density, 0, 1 );

    d3dcrs( D3DRS_FOGDENSITY, d3dvar(d) );
}
gmxd d3d_set_fog_color   ( double col      )
{
    // Set fog colour.
    d3dcrs( D3DRS_FOGCOLOR, col_d3d( (int) col, 0.0 ) );
}
gmxd d3d_set_fog_start   ( double dist     )
{
    // Set fog start.
    float d = (float) dist;
    d3dcrs( D3DRS_FOGSTART, d3dvar(d) );
}
gmxd d3d_set_fog_end     ( double dist     )
{
    // Set fog end.
    float d = (float) dist;
    d3dcrs( D3DRS_FOGEND, d3dvar(d) );
}

// Points
gmxd d3d_set_point_size       ( double size  )
{
    // Set drawing size for point primitives.
    float x = (float) clamp(size,1.0,d3dcaps.MaxPointSize);

    d3dcrs( D3DRS_POINTSIZE, d3dvar(x) );
}
gmxd d3d_set_point_size_min   ( double size  )
{
    // Set size clamp, useful for scaled points in 3D mode.  Defaults to 1.
    float x = (float) clamp(size,1.0,d3dcaps.MaxPointSize);

    d3dcrs( D3DRS_POINTSIZE_MIN, d3dvar(x) );
}
gmxd d3d_set_point_size_max   ( double size  )
{
    // Set size clamp.  Defaults to 64.
    float x = (float) clamp(size,1.0,d3dcaps.MaxPointSize);

    d3dcrs( D3DRS_POINTSIZE_MAX, d3dvar(x) );
}
gmxd d3d_set_point_scale      ( double state )
{
    // Enable point scaling.  This scales points based on their distance from the camera.
    d3dcrs( D3DRS_POINTSCALEENABLE, (state > 0.0) );
}
gmxd d3d_set_point_scale_coef ( double coef1, double coef2, double coef3 )
{
    // Configure point scaling formula.  Defaults to (1,0,0).  The formula is: size * sqrt(1/( ceof1 + (coef2*distancetocamera) + (coef3*sqr(distancetocamera)) ))
    dword a, b, c;
    float ca,cb,cc;

    ca = (float) abs(coef1);
    cb = (float) abs(coef2);
    cc = (float) abs(coef3);
    
    a  = d3drs( D3DRS_POINTSCALE_A, d3dvar(ca) );
    b  = d3drs( D3DRS_POINTSCALE_B, d3dvar(cb) );
    c  = d3drs( D3DRS_POINTSCALE_C, d3dvar(cc) );
    return (double) ( a == D3D_OK and b == D3D_OK and c == D3D_OK );
}
gmxd d3d_set_point_sprite     ( double state )
{
    // When enabled, this causes point primitives to be drawn with textures applied.
    d3dcrs( D3DRS_POINTSPRITEENABLE, (state > 0.0) );
}

// Render control
gmxd d3d_set_mask        ( double r, double g, double b, double a )
{
    // Set colour writemask.  You can enable/disable writing of each channel independently.  All enabled by default.
    if (!(d3dcaps.PrimitiveMiscCaps & D3DPMISCCAPS_COLORWRITEENABLE))
    { return gfalse; }

    dword mask [4];
    mask[0] = (r > 0.0) ? D3DCOLORWRITEENABLE_RED   : 0;
    mask[1] = (g > 0.0) ? D3DCOLORWRITEENABLE_GREEN : 0;
    mask[2] = (b > 0.0) ? D3DCOLORWRITEENABLE_BLUE  : 0;
    mask[3] = (a > 0.0) ? D3DCOLORWRITEENABLE_ALPHA : 0;

    d3dcrs( D3DRS_COLORWRITEENABLE, mask[0] | mask[1] | mask[2] | mask[3] );
}
gmxd d3d_set_zwrite      ( double state )
{
    // Set z-buffer writing.  Enabled by default.  Disabling it prevents overwriting of the z-buffer.
    d3dcrs( D3DRS_ZWRITEENABLE, (state > 0.0) );
}
gmxd d3d_set_alphatest   ( double value, double mode )
{
    // Prevents drawing of pixels that don't meet the given alpha criteria.  Use the cmp_ constants.  Pass -1 as value to disable alpha testing.
    dword a,b,c;
    if (value < 0.0)
         { d3dcrs( D3DRS_ALPHATESTENABLE, false ); }
    else { a = d3drs( D3DRS_ALPHATESTENABLE, true  );
           b = d3drs( D3DRS_ALPHAREF,        (dword) clamp(value,0x00000000,0x000000FF) );
           c = d3drs( D3DRS_ALPHAFUNC,       (dword) mode );}

    return (a==D3D_OK and b==D3D_OK and c==D3D_OK);
}
gmxd d3d_set_ztest       ( double mode )
{
    // Prevents drawing of pixels that don't meet the given depth criteria.  Use the cmp_ constants.  Defaults to <=.  The other value is the current z-buffer value.
    if (!(d3dcaps.RasterCaps & D3DPRASTERCAPS_ZTEST ))
    { return gfalse; }

    d3dcrs( D3DRS_ZFUNC, (dword) mode );
}
gmxd d3d_set_zbias       ( double bias )
{
    // Offsets the drawing depth, allowing polygons with the same positions to be drawn without z-fighting artifacts.  Useful for shadows, decals, etc.   Integer 0-16, defaults to 0.  Higher values appear in front of lower ones.
    d3dcrs( D3DRS_ZBIAS, (uint) floor(clamp(bias,0,16)) );
}
gmxd d3d_set_fillmode    ( double mode )
{
    // Set how D3D renders polygons: point, wireframe or solid.  Use d3d_fillmode_ constants.  Defaults to solid.
    d3dcrs( D3DRS_FILLMODE, (dword) mode );
}
gmxd d3d_set_normal_auto ( double state )
{
    // Automatically normalises vectors when rendering.  Should solve problems with models changing brightness when scaled.
    d3dcrs( D3DRS_NORMALIZENORMALS, (state > 0.0) );
}

// Primitives
gmxd d3d_primitive_begin_ext  ( double primitive, double textured )
{
    // Begin drawing an extended primitive.
    vbuff_c       = 0;
    vbuff_autoinc = (textured < 0.5);

    // Zero the buffer.
    SecureZeroMemory( (PVOID) vbuff_int, vb_bytes );

    switch ((int)primitive)
    {
        case 1: { vbuff_prim = D3DPT_POINTLIST;     } break;
        case 2: { vbuff_prim = D3DPT_LINELIST;      } break;
        case 3: { vbuff_prim = D3DPT_LINESTRIP;     } break;
        case 4: { vbuff_prim = D3DPT_TRIANGLELIST;  } break;
        case 5: { vbuff_prim = D3DPT_TRIANGLESTRIP; } break;
        case 6: { vbuff_prim = D3DPT_TRIANGLEFAN;   } break;
        default: { return gerror; } break;
    }

    return gtrue;
}
gmxd d3d_vertex_ext           ( double x, double y, double z, double nx, double ny, double nz, double col, double alpha, double speccol, double specalpha )
{
    // Position, normal, diffuse/specular colour and alpha.
    vbuff_int[vbuff_c].x  = (float) x;
    vbuff_int[vbuff_c].y  = (float) y;
    vbuff_int[vbuff_c].z  = (float) z;
    vbuff_int[vbuff_c].nx = (float) nx;
    vbuff_int[vbuff_c].ny = (float) ny;
    vbuff_int[vbuff_c].nz = (float) nz;
    vbuff_int[vbuff_c].c  = col_d3d( (int) col,     alpha     );
    vbuff_int[vbuff_c].s  = col_d3d( (int) speccol, specalpha );

    if (vbuff_autoinc)
    { vbuff_c++; }

    return gtrue;
}
gmxd d3d_vertex_ext_tex       ( double stage, double xtex, double ytex )
{
    // Set vertex texture coordinates.  There are eight sets indexed 0-7, one for each texture stage.
    uint ind = (uint) (clamp(stage,0.0,7.0)*2.0);

    vbuff_int[vbuff_c].uv[ind]   = (float) xtex;
    vbuff_int[vbuff_c].uv[ind+1] = (float) ytex;

    return gtrue;
}
gmxd d3d_vertex_ext_next      ()
{
    // Call when finished with the current vertex to start defining the next one.  Only required for textured primitives.
    if (vbuff_autoinc)
    { complain("d3d_vertex_ext_next() is called automatically for untextured extended primitives.");}

    vbuff_c++;

    return gtrue;
}
gmxd d3d_primitive_end_ext    ()
{
    // Draw the primitive.
    BYTE* bytep;
    uint  count = (vbuff_c+1);
    uint  size  = (count*sizeof(vert_ext));
    uint  prims;

    if (count < 1)
    { return gerror; }

    switch (vbuff_prim)
    {
     case D3DPT_POINTLIST:     { prims = snap_low(count,2);           } break;
     case D3DPT_LINELIST:      { prims = snap_low(count,2)/2;         } break;
     case D3DPT_LINESTRIP:     { prims = snap_low(count,2);           } break;
     case D3DPT_TRIANGLELIST:  { prims = snap_low(count,3)/2;         } break;
     case D3DPT_TRIANGLESTRIP: { prims = uint(snap_low(count,2)/1.5); } break;
     case D3DPT_TRIANGLEFAN:   { prims = uint(snap_low(count,2)/1.5); } break;
     default: { return gerror; } break;
    }


    if (prims < 1)
    { return gerror; }

    vbuff_d3d->Lock( 0, size, &bytep, 0 );
    memcpy( bytep, vbuff_int, size ); // Copy only used part to conserve bandwidth
    vbuff_d3d->Unlock();

    
    dword a,b,c;
    a = d3ddev->SetStreamSource( 0, vbuff_d3d, sizeof(vert_ext) );
    
    if (!vbuff_usevs)
         { b = d3ddev->SetVertexShader( fvf_ext ); }
    else { b = D3D_OK; }

    c = d3ddev->DrawPrimitive( vbuff_prim, 0, prims );

    return (double) ((a==D3D_OK) and (b==D3D_OK) and (c==D3D_OK));
}

gmxd draw_primitive_begin_ext ( double primitive, double textured )
{
    // 2D equivalent.
    return d3d_primitive_begin_ext( primitive, textured );
}
gmxd draw_vertex_ext          ( double x, double y, double col, double alpha, double speccol, double specalpha )
{
    // 2D equivalent.
    // The buffer is zeroed by begin_ext; no need to set the 3D stuff here.

    vbuff_int[vbuff_c].x = (float) x;
    vbuff_int[vbuff_c].y = (float) y;

    vbuff_int[vbuff_c].c = col_d3d( (int) col,     alpha     );
    vbuff_int[vbuff_c].s = col_d3d( (int) speccol, specalpha );

    if (vbuff_autoinc)
    { vbuff_c++; }

    return gtrue;
}
gmxd draw_vertex_ext_tex      ( double stage, double xtex, double ytex )
{
    // 2D equivalent.
    return d3d_vertex_ext_tex( stage, xtex, ytex );
}
gmxd draw_vertex_ext_next     ()
{
    // 2D equivalent.
    return d3d_vertex_ext_next();
}
gmxd draw_primitive_end_ext   ()
{
    // 2D equivalent.
    return d3d_primitive_end_ext();
}
























