#version 120
varying vec3 position;
varying vec3 n, e, l;
varying vec3 worldView;
varying vec3 worldNormal;

varying vec4 shadowCoord1;
varying vec4 shadowCoord2;
varying vec4 shadowCoord3;

uniform sampler2D texture;
uniform sampler2D normalMap;
uniform bool hasNormalMap;

uniform sampler2D roughnessMap;
uniform bool hasRoughnessMap;

uniform sampler2D metallicMap;
uniform bool hasMetallicMap;

uniform sampler2D heightMap;
uniform bool hasHeightMap;

uniform samplerCube envMap;
uniform mat4 viewMatrix;

uniform sampler2DShadow shadowMap1;
uniform sampler2DShadow shadowMap2;
uniform sampler2DShadow shadowMap3;

uniform vec2 shadowMapSize; 

uniform bool usePCF;
uniform bool drawCascades;

const float defaultRoughness = 0.4;
const float defaultMetallic = 0.01;

#define PI 3.14159265

const float parallaxScale = 0.03;
const float parallaxBias = -0.01;

mat3 cotangentFrame(vec3 N, vec3 p, vec2 uv)
{
    vec3 dp1 = dFdx(p);
    vec3 dp2 = dFdy(p);
    vec2 duv1 = dFdx(uv);
    vec2 duv2 = dFdy(uv);
    vec3 dp2perp = cross(dp2, N);
    vec3 dp1perp = cross(N, dp1);
    vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;
    vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;
    float invmax = inversesqrt(max(dot(T, T), dot(B, B)));
    return mat3(T * invmax, B * invmax, N);
}

float shadowLookup(sampler2DShadow depths, vec4 coord, vec2 offset)
{
    vec2 texelSize = 1.0 / shadowMapSize;
    vec2 v = offset * texelSize * coord.w;
    float z = shadow2DProj(depths, coord + vec4(v.x, v.y, 0.0, 0.0)).z;
    return z;
}

float weight(vec4 tc)
{
    vec2 proj = vec2(tc.x / tc.w, tc.y / tc.w);
    proj = (1.0 - abs(proj * 2.0 - 1.0)) * 8.0;
    proj = clamp(proj, 0.0, 1.0);
    return min(proj.x, proj.y);
}

float pcf(sampler2DShadow depths, vec4 coord, float radius)
{
    float s = 0.0;
    float x, y;
	for (y = -radius ; y < radius ; y += 1.0)
	for (x = -radius ; x < radius ; x += 1.0)
    {
	    s += shadowLookup(depths, coord, vec2(x, y));
    }
	s /= radius * radius * 4.0;
    return s;
}

float beckmann(float NH, float roughness)
{
    float NH_sq = NH * NH;
    float NH_sq_r = 1.0 / (NH_sq * roughness * roughness);
    float roughness_exp = (NH_sq - 1.0) * NH_sq_r;
    return exp(roughness_exp)*NH_sq_r/(4.0*NH_sq);
}
    
float cookTorrance(vec3 lightDirection, vec3 viewDirection, vec3 surfaceNormal, float roughness, float fresnel)
{
    float VdotN = max(dot(viewDirection, surfaceNormal), 0.0);
    float LdotN = max(dot(lightDirection, surfaceNormal), 0.0);
    vec3 H = normalize(lightDirection + viewDirection);
    float NdotH = max(dot(surfaceNormal, H), 0.0);
    float VdotH = max(dot(viewDirection, H), 0.000001);
    float LdotH = max(dot(lightDirection, H), 0.000001);
    float G = 2.0 * NdotH / max(VdotH, 0.000001);
    G = min(1.0, G * min(VdotN, LdotN));
    float D = beckmann(NdotH, roughness);
    float F = fresnel + pow(1.0 - VdotH, 5.0) * (1.0 - fresnel);
    return  G * F * D / max(PI * VdotN, 0.000001);
}

float blinnPhong(vec3 L, vec3 E, vec3 N, float roughness)
{
    vec3 H = normalize(L + E);
    float NH = dot(N, H);
    return pow(max(NH, 0.0), 64.0 * (1.0 - roughness));
}

const vec3 c_white = vec3(1.0, 1.0, 1.0);
const vec3 c_red   = vec3(1.0, 0.0, 0.0);
const vec3 c_green = vec3(0.0, 1.0, 0.0);
const vec3 c_blue  = vec3(0.0, 0.0, 1.0);

const vec3 sunColor = vec3(0.9, 0.6, 0.3) * 2.0;
    
void main()
{
    float s1, s2, s3;
    if (usePCF)
    {
        s1 = pcf(shadowMap1, shadowCoord1, 3.0);
        s2 = pcf(shadowMap2, shadowCoord2, 2.0);
        s3 = pcf(shadowMap3, shadowCoord3, 2.0);
    }
    else
    {
        s1 = shadow2DProj(shadowMap1, shadowCoord1).z;
        s2 = shadow2DProj(shadowMap2, shadowCoord2).z;
        s3 = shadow2DProj(shadowMap3, shadowCoord3).z;
    }

    float w1 = weight(shadowCoord1);
    float w2 = weight(shadowCoord2);
    float w3 = weight(shadowCoord3);
    
    s3 = mix(1.0, s3, w3);
    s2 = mix(s3, s2, w2);
    s1 = mix(s2, s1, w1);
    
    // remove this if you don't need cascade visualization
    vec3 c3 = mix(c_white, c_blue, w3);
    vec3 c2 = mix(c3, c_green, w2);
    vec3 c1 = mix(c2, c_red, w1);
    
    vec3 N = normalize(n);
    vec3 E = normalize(e);
    vec3 L = normalize(l);
    float roughness = defaultRoughness;
    float metallic = defaultMetallic;
    
    vec2 texCoords = (gl_TexCoord[0] * gl_TextureMatrix[0]).st;
    
    vec3 albedo = texture2D(texture, texCoords).rgb;
    
    mat3 TBN = cotangentFrame(N, position, texCoords);
    
    if (hasHeightMap)
    {
        float height = texture2D(heightMap, texCoords).r;
        height = height * parallaxScale + parallaxBias;
        vec3 Ee = normalize(E * TBN);
        texCoords = texCoords + (height * Ee.xy);
    }
    
    if (hasRoughnessMap)
        roughness = max(texture2D(roughnessMap, texCoords).r, 0.001);
        
    if (hasMetallicMap)
        metallic = max(texture2D(metallicMap, texCoords).r, 0.001);
    
    if (hasNormalMap)
    {
        vec3 tN = normalize(texture2D(normalMap, texCoords).rgb * 2.0 - 1.0);
        N = normalize(TBN * tN);
    }
    
    vec3 worldN = normalize(worldNormal);
    vec3 worldR = reflect(normalize(worldView), worldN);
    worldR.x = -worldR.x;
    
    int diffLod = int(8.0);
    float roughnessFloor = floor(roughness * 8.0);
    int specLod1 = int(roughnessFloor);
    int specLod2 = int(ceil(roughness * 8.0));
    const float oneOver8 = 1.0 / 8.0;
    vec3 envDiff = textureCubeLod(envMap, worldN, diffLod).rgb;
    vec3 envSpecMin = textureCubeLod(envMap, worldR, specLod1).rgb;
    vec3 envSpecMax = textureCubeLod(envMap, worldR, specLod2).rgb;
    vec3 envSpec = mix(envSpecMin, envSpecMax, roughness - roughnessFloor * oneOver8);
    
    float dirDiffBrightness = clamp(dot(N, L), 0.0, 1.0);
    float dirSpecBrightness = clamp(cookTorrance(L, E, N, roughness, 0.3), 0.0, 1.0);
    
    vec3 pointDiffSum = vec3(0.0, 0.0, 0.0);
    vec3 pointSpecSum = vec3(0.0, 0.0, 0.0);
    
    const int maxNumLights = 2;
    
    vec3 directionToLight;
    float attenuation;
    
    for (int i = 0; i < maxNumLights; i++)
    {
        float spotEffect = 1.0;        
        if (gl_LightSource[i].position.w > 0.0)
        {
            vec3 positionToLightSource = vec3(gl_LightSource[i].position.xyz - position);
            float distanceToLight = length(positionToLightSource);
            directionToLight = normalize(positionToLightSource);
            attenuation =
                      gl_LightSource[i].constantAttenuation /
              ((1.0 + gl_LightSource[i].linearAttenuation * distanceToLight) *
               (1.0 + gl_LightSource[i].quadraticAttenuation * distanceToLight * distanceToLight));
               
            if (gl_LightSource[i].spotCutoff <= 90.0)
            {
                float spotCos = dot(directionToLight, normalize(gl_LightSource[i].spotDirection));
                spotEffect = clamp(pow(spotCos, gl_LightSource[i].spotExponent) * step(gl_LightSource[i].spotCosCutoff, spotCos), 0.0, 1.0);
            }
        }
        else
        {
            directionToLight = gl_LightSource[i].position.xyz;
            attenuation = 1.0;
        }

        pointDiffSum += (gl_LightSource[i].ambient.rgb + gl_LightSource[i].diffuse.rgb * clamp(dot(N, directionToLight), 0.0, 1.0)) * attenuation;
        pointSpecSum += gl_LightSource[i].specular.rgb * clamp(cookTorrance(directionToLight, E, N, roughness, 0.3), 0.0, 1.0) * attenuation;
    }
    
    float fogDistance = gl_FragCoord.z / gl_FragCoord.w;
    float fogFactor = clamp((gl_Fog.end - fogDistance) / (gl_Fog.end - gl_Fog.start), 0.0, 1.0);
    
    vec3 diffLight = envDiff + pointDiffSum + sunColor * dirDiffBrightness * s1;
    vec3 specLight = envSpec + pointSpecSum + sunColor * dirSpecBrightness * s1;
    
    vec3 diffReflectivity = albedo * (1.0 - metallic);

    vec3 f0 = mix(vec3(0.03, 0.03, 0.03), albedo, metallic);
    
    float fresnel = pow(1.0 - max(0.0, dot(N, E)), 5.0);
    vec3 specReflectivity = f0 + (1.0 - f0) * fresnel * mix(1.0, 0.75, roughness) * (1.0 - roughness);
    
    diffReflectivity *= 1.0 - specReflectivity;
    
    vec3 objColor = diffLight * diffReflectivity + specLight * specReflectivity + (specLight - envSpec) * (1.0 - roughness);
    
    vec3 color = drawCascades? c1 * s1 : mix(gl_Fog.color.rgb, objColor, fogFactor);
    
    gl_FragColor = vec4(color, 1.0);
}
