#version 120
varying vec3 position;
varying vec3 normal;
varying vec3 tangent;
varying vec3 binormal;
varying vec3 eyeTan;
varying vec3 eye;
varying vec3 worldView;

uniform samplerCube texture0;
uniform sampler2D texture1;
uniform mat4 viewMatrix;
uniform vec4 rippleTimes;

#define PI 3.1415926535897932384626433832795

const vec4 waterColor = vec4(0.0, 0.05, 0.05, 1.0);
const float textureScale = 0.2;
const float rainIntensity = 1.0;

vec3 computeRipple(vec2 uv, float currentTime, float weight)
{
    vec4 ripple = texture2D(texture1, uv);
    ripple.yz = ripple.yz * 2.0 - 1.0;
    float dropFrac = fract(ripple.w + currentTime);
    float timeFrac = dropFrac - 1.0 + ripple.x;
    float dropFactor = clamp(0.2 + weight * 0.8 - dropFrac, 0.0, 1.0);
    float finalFactor = dropFactor * ripple.x * sin(clamp(timeFrac * 9.0, 0.0, 3.0) * PI);
    return vec3(ripple.yz * finalFactor * 0.35, 1.0);
}
    
void main()
{
    vec3 dirToLight = normalize(gl_LightSource[0].position.xyz);
    vec3 eyeN = normalize(normal);
    vec3 eyeT = normalize(tangent);
    vec3 eyeB = normalize(binormal);
    vec3 L = vec3(dot(dirToLight, eyeT),
                  dot(dirToLight, eyeB),
                  dot(dirToLight, eyeN));
   
    vec2 uv = gl_TexCoord[0].st * textureScale;
    
    vec4 weights = rainIntensity - vec4(0, 0.25, 0.5, 0.75);
    weights = clamp(weights * 4.0, 0.0, 1.0);
    
    vec3 ripple1 = computeRipple(uv + vec2( 0.25, 0.0), rippleTimes.x, weights.x);
    vec3 ripple2 = computeRipple(uv + vec2(-0.55, 0.3), rippleTimes.y, weights.y);
    vec3 ripple3 = computeRipple(uv + vec2( 0.6, 0.85), rippleTimes.z, weights.z);
    vec3 ripple4 = computeRipple(uv + vec2( 0.5,-0.75), rippleTimes.w, weights.w);

    vec4 Z = mix(vec4(1.0, 1.0, 1.0, 1.0), vec4(ripple1.z, ripple2.z, ripple3.z, ripple4.z), weights);
    vec3 Normal = vec3(
                   weights.x * ripple1.xy +
                   weights.y * ripple2.xy + 
                   weights.z * ripple3.xy + 
                   weights.w * ripple4.xy, 
                   Z.x * Z.y * Z.z * Z.w);                             
    vec3 N = normalize(Normal);

    vec3 E = normalize(eyeTan);
    vec3 worldNormal = transpose(mat3(viewMatrix)) * N.xzy;
    vec3 R = reflect(normalize(worldView), worldNormal);
    R = normalize(vec3(-R.x, R.y, -R.z));

    vec3 H = normalize(L + E);

    float fresnel = 1.0 - max(dot(eye, eyeN), 0.0);

    float diffuse = clamp(dot(N, L), 0.0, 1.0);
    
    vec4 tex = textureCube(texture0, R); 
    vec4 reflectionTerm = tex * fresnel + waterColor * diffuse * (1.0 - fresnel);
   
    vec4 col =
     reflectionTerm +
     diffuse * vec4(0.07, 0.07, 0.07, 1.0);
     
    col.a = max(0.9, fresnel);

    gl_FragColor = col;
}
